package ru.yandex.direct.jooqmapper.jsonread;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Objects.requireNonNull;

@ParametersAreNonnullByDefault
public class JooqJsonReader<M extends Model> {
    private final ObjectMapper mapper = new ObjectMapper();
    private final ImmutableMap<ModelProperty<? super M, ?>, JsonReader<?>> jsonReaders;
    private final ImmutableSet<ModelProperty<? super M, ?>> readableModelProperties;

    public JooqJsonReader(Map<ModelProperty<? super M, ?>, JsonReader<?>> jsonReaders) {
        this.jsonReaders = ImmutableMap.copyOf(requireNonNull(jsonReaders, "readers map is required"));
        checkArgument(!jsonReaders.isEmpty(), "readers map must not be empty");
        this.readableModelProperties = ImmutableSet.copyOf(jsonReaders.keySet());
    }

    public ImmutableSet<ModelProperty<? super M, ?>> getReadableModelProperties() {
        return readableModelProperties;
    }

    public ImmutableMap<ModelProperty<? super M, ?>, JsonReader<?>> getJsonReaders() {
        return jsonReaders;
    }

    public M fromDb(Record record,M model) {
        return fromDb(record, model, ImmutableSet.copyOf(record.fields()));
    }

    public M fromDb(Record record, M model, Set<Field> availableFields) {
        boolean someDataWasRead = false;

        // Выбираем список полей таблицы, из которых читают зарегистрированные jsonReaders
        Set<Field<?>> jsonFields = StreamEx.ofValues(jsonReaders)
                .<Field<?>>map(JsonReader::getRequiredDatabaseField)
                .filter(availableFields::contains)
                .toImmutableSet();

        // Парсим json из этих полей таблицы в кеш
        Cache<Field<?>, JsonNode> cache = parseFieldsToCache(record, jsonFields);

        // Применяем все ридеры
        for (Map.Entry<ModelProperty<? super M, ?>, JsonReader<?>> propAndReader : jsonReaders.entrySet()) {
            ModelProperty modelProperty = propAndReader.getKey();
            JsonReader reader = propAndReader.getValue();
            Field<?> field = reader.getRequiredDatabaseField();

            JsonNode rootJsonNode = cache.getIfPresent(field);
            if (rootJsonNode != null) {
                JsonNode jsonNode = rootJsonNode.get(reader.getJsonPath());
                modelProperty.set(model, reader.read(jsonNode));
                someDataWasRead = true;
            }
        }

        checkState(someDataWasRead, "there is not enough data in record to read at least one model property");

        return model;
    }

    private Cache<Field<?>, JsonNode> parseFieldsToCache(Record record, Set<Field<?>> fields) {
        Cache<Field<?>, JsonNode> cache = CacheBuilder.newBuilder().build(new CacheLoader<>() {
            @Override
            public JsonNode load(Field<?> key) throws Exception {
                return null;
            }
        });

        for (Field field : fields) {
            Object o = record.get(field);
            if (o instanceof String) {
                JsonNode jsonNode = parseJson((String) o);
                if (jsonNode != null) {
                    cache.put(field, jsonNode);
                }
            }
        }
        return cache;
    }

    private JsonNode parseJson(String json) {
        try {
            return mapper.readTree(json);
        } catch (IOException e) {
            return null;
        }
    }

    public M fromDb(Record record, M model, List<ModelProperty<? super M, ?>> modelProperties) {
        checkArgument(!modelProperties.isEmpty(), "model properties to read must be provided");

        Set<Field> availableFields = ImmutableSet.copyOf(record.fields());

        // Подготовка для json полей
        // Получаем список полей таблицы, в которых хранится json для инициализации переданных modelProperty
        Set<Field<?>> jsonFields = StreamEx.of(modelProperties)
                .map(modelProperty -> {
                    JsonReader<?> jsonReader = jsonReaders.get(modelProperty);
                    checkArgument(jsonReader != null,
                            "there are no registered readers for model property " + modelProperty);
                    return jsonReader;
                })
                .<Field<?>>map(JsonReader::getRequiredDatabaseField)
                .filter(availableFields::contains)
                .toSet();

        Cache<Field<?>, JsonNode> cache = parseFieldsToCache(record, jsonFields);

        Set<? extends ModelProperty<? super M, ?>> jsonModelProperties = EntryStream.of(jsonReaders)
                .filterValues(reader -> jsonFields.contains(reader.getRequiredDatabaseField()))
                .map(Map.Entry::getKey)
                .toSet();

        for (ModelProperty modelProperty : jsonModelProperties) {
            JsonReader jsonReader = jsonReaders.get(modelProperty);
            Field<?> field = jsonReader.getRequiredDatabaseField();

            checkArgument(availableFields.contains(field),
                    "there is not enough data in record to read model property " + modelProperty);

            JsonNode rootJsonNode = cache.getIfPresent(field);
            if (rootJsonNode != null) {
                JsonNode jsonNode = rootJsonNode.get(jsonReader.getJsonPath());
                modelProperty.set(model, jsonReader.read(jsonNode));
            }
        }

        return model;
    }

    /**
     * @param modelProperties список свойств модели.
     * @return список полей БД, которые необходимо прочитать для заполнения указанных свойств модели.
     */
    public Set<Field<?>> getFieldsToRead(Collection<ModelProperty<?, ?>> modelProperties) {
        return StreamEx.of(modelProperties)
                .<Field<?>>map(modelProperty -> {
                    JsonReader<?> jsonReader = jsonReaders.get(modelProperty);
                    checkArgument(jsonReader != null,
                            "there are no registered readers for model property " + modelProperty);
                    return jsonReader.getRequiredDatabaseField();
                })
                .toSet();
    }

    /**
     * @return список полей БД, которые необходимо прочитать для заполнения
     * всех свойств модели, которые умеет читать данный маппер.
     */
    public Set<Field<?>> getFieldsToRead() {
        return StreamEx.ofValues(jsonReaders)
                .<Field<?>>map(JsonReader::getRequiredDatabaseField)
                .toImmutableSet();
    }

    /**
     * @see #canReadAtLeastOneProperty(Collection)
     */
    public boolean canReadAtLeastOneProperty(Field<?>[] fieldsCandidates) {
        Set<? super Field<?>> fieldsCandidatesSet = ImmutableSet.copyOf(fieldsCandidates);
        return canReadAtLeastOneProperty(fieldsCandidatesSet);
    }

    public boolean canReadAtLeastOneProperty(Collection<? super Field<?>> fieldsCandidates) {
        Set<? super Field<?>> fieldsCandidatesSet = ImmutableSet.copyOf(fieldsCandidates);
        return jsonReaders.values()
                .stream()
                .map(JsonReader::getRequiredDatabaseField)
                .anyMatch(fieldsCandidatesSet::contains);
    }
}
