package ru.yandex.direct.jooqmapper.jsonread;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import org.jooq.Field;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReader;

import static ru.yandex.direct.jooqmapper.jsonwrite.JsonWriterBuilders.LOCAL_DATE_TIME_FORMATTER;

/**
 * Для построения {@link JooqReader} используйте {@link JooqJsonReaderBuilder},
 * а для построения {@link JooqMapper} - {@link JooqMapperBuilder}.
 */
public final class JsonReaderBuilders {

    private JsonReaderBuilders() {
    }

    public static <T extends Enum<T>> T jsonNodeToEnum(JsonNode node, Class<T> enumClass) {
        return isInvalidNode(node) ? null : Enum.valueOf(enumClass, node.asText());
    }

    public static Boolean jsonNodeToBoolean(JsonNode node) {
        return isInvalidNode(node) ? null : node.asBoolean();
    }

    @Nullable
    public static Integer jsonNodeToInteger(JsonNode node) {
        return isInvalidNode(node) ? null : node.asInt();
    }

    @Nullable
    public static Long jsonNodeToLong(JsonNode node) {
        return isInvalidNode(node) ? null : node.asLong();
    }

    public static String jsonNodeToString(JsonNode node) {
        return isInvalidNode(node) ? null : node.asText();
    }

    public static Boolean longJsonNodeToBoolean(JsonNode node) {
        return isInvalidNode(node) ? null : node.asLong() == 1L;
    }

    public static LocalDate stringJsonNodeToLocalDate(JsonNode node) {
        return isInvalidNode(node) ? null : LocalDate.parse(node.textValue(), DateTimeFormatter.ISO_LOCAL_DATE);
    }

    public static LocalDateTime stringJsonNodeToLocalDateTime(JsonNode node) {
        return isInvalidNode(node) ? null : LocalDateTime.parse(node.textValue(), LOCAL_DATE_TIME_FORMATTER);
    }

    public static BigDecimal readBigDecimalAnyOfType(JsonNode node) {
        return parseBigDecimal(node);
    }

    public static JsonReader1Builder<Boolean> fromJsonToBoolean(Field<String> field, String jsonPath) {
        return JsonReaderBuilders.fromFieldAndPath(field, jsonPath).by(JsonReaderBuilders::jsonNodeToBoolean);
    }

    public static JsonReader1Builder<Long> fromJsonToLong(Field<String> field, String jsonPath) {
        return JsonReaderBuilders.fromFieldAndPath(field, jsonPath).by(node -> node != null ? node.asLong() : null);
    }

    public static JsonReader1Builder<Integer> fromJsonToInteger(Field<String> field, String jsonPath) {
        return JsonReaderBuilders.fromFieldAndPath(field, jsonPath).by(JsonReaderBuilders::jsonNodeToInteger);
    }

    public static JsonReader1Builder<String> fromJsonToString(Field<String> field, String jsonPath) {
        return JsonReaderBuilders.fromFieldAndPath(field, jsonPath).by(JsonReaderBuilders::jsonNodeToString);
    }

    public static <T extends Enum<T>> JsonReader1Builder<T> fromJsonToEnum(Field<String> field, String jsonPath,
                                                                           Class<T> enumClass) {
        return JsonReaderBuilders.fromFieldAndPath(field, jsonPath).by(x -> jsonNodeToEnum(x, enumClass));
    }

    public static JsonReader1Builder.JsonReader1WithFieldStep fromFieldAndPath(Field<String> field, String jsonPath) {
        return JsonReader1Builder.fromFieldAndPath(field, jsonPath);
    }

    private static boolean isInvalidNode(JsonNode node) {
        return node == null || node.isMissingNode() || node.isNull();
    }

    @Nullable
    private static BigDecimal parseBigDecimal(JsonNode node) {
        if (isInvalidNode(node)) {
            return null;
        } else if (node.getNodeType() == JsonNodeType.NUMBER) {
            return node.decimalValue();
        } else if (node.getNodeType() == JsonNodeType.STRING) {
            return new BigDecimal(node.textValue());
        } else {
            throw new IllegalArgumentException("Unexpected node " + node);
        }
    }
}
