package ru.yandex.direct.jooqmapper.jsonwrite;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.jooqmapper.write.PropertyValues;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Вычисляет значение для записи в JSON.
 * Умеет отдавать свойства модели, которые будут прочитаны для записи в JSON.
 *
 * @param <M> тип модели.
 */
@ParametersAreNonnullByDefault
public class DefaultJsonWriter<M extends Model> implements JsonWriter<M> {

    private final ImmutableSet<ModelProperty<? super M, ?>> properties;
    private final Function<PropertyValues<M>, JsonNode> writerFunction;
    private final String path;

    DefaultJsonWriter(Set<ModelProperty<? super M, ?>> properties,
                      String path,
                      Function<PropertyValues<M>, JsonNode> writerFunction) {
        this.properties = ImmutableSet.copyOf(requireNonNull(properties, "properties set is required"));
        this.writerFunction = requireNonNull(writerFunction, "writer function is required");
        this.path = requireNonNull(path, "json path is required");
        checkArgument(!properties.isEmpty(), "properties set must not be empty");
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getProperties() {
        return properties;
    }

    @Override
    public JsonNode write(M model) {
        PropertyValues<M> propertyValues = new PropertyValues<>(properties, model);
        return writerFunction.apply(propertyValues);
    }

    @Override
    public String getJsonPath() {
        return path;
    }
}
