package ru.yandex.direct.jooqmapper.jsonwrite;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

@ParametersAreNonnullByDefault
public class JooqJsonWriter<M extends Model> {

    private final ImmutableMultimap<TableField<? extends Record, ?>, JsonWriter<? super M>> writerMap;

    private final ImmutableSet<ModelProperty<? super M, ?>> writableModelProperties;

    private final ObjectMapper objectMapper = new ObjectMapper();

    public JooqJsonWriter(Multimap<TableField<? extends Record, ?>, JsonWriter<? super M>> writerMap) {
        checkArgument(!writerMap.isEmpty(), "writers map must not be empty");
        this.writerMap = ImmutableMultimap.copyOf(requireNonNull(writerMap, "writers map is required"));
        this.writableModelProperties = ImmutableSet.copyOf(
                StreamEx.of(writerMap.values()).map(JsonWriter::getProperties).flatMap(Collection::stream).toSet()
        );
    }

    ImmutableMultimap<TableField<? extends Record, ?>, JsonWriter<? super M>> getWriters() {
        return writerMap;
    }

    public ImmutableSet<ModelProperty<? super M, ?>> getWritableModelProperties() {
        return writableModelProperties;
    }

    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(M model, Table<R> table) {
        Set<TableField<?, ?>> fields = StreamEx.of(writerMap.keySet())
                .filter(tableField -> tableField.getTable().equals(table))
                .toSet();

        Map<TableField<R, ?>, String> map = new HashMap<>();

        for (TableField field : fields) {
            JsonNode rootNode = objectMapper.createObjectNode();
            ImmutableCollection<JsonWriter<? super M>> jsonWriters = writerMap.get(field);
            for (JsonWriter<? super M> jsonWriter : jsonWriters) {
                String jsonPath = jsonWriter.getJsonPath();
                JsonNode jsonNode = jsonWriter.write(model);
                // Skip null values
                if (jsonNode != null && !jsonNode.isNull()) {
                    ((ObjectNode) rootNode).set(jsonPath, jsonNode);
                }
            }
            try {
                map.put(field, objectMapper.writeValueAsString(rootNode));
            } catch (JsonProcessingException e) {
                throw new RuntimeException();
            }
        }
        return map;
    }

    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(
            M model,
            Set<ModelProperty<? super M, ?>> requestedProperties, Table<R> table) {

        Predicate<Collection<JsonWriter<? super M>>> atLeastOneRelevantWriterPredicate = writers -> StreamEx.of(writers)
                .findAny(writer -> !Sets.intersection(requestedProperties, writer.getProperties()).isEmpty())
                .isPresent();

        Set<? extends TableField<R, ?>> tableFields = EntryStream.of(writerMap.asMap())
                .filterKeys(tableField -> tableField.getTable().equals(table))
                .filterValues(atLeastOneRelevantWriterPredicate)
                .map(entry -> (TableField<R, ?>) entry.getKey())
                .toImmutableSet();

        Map<TableField<R, ?>, String> map = new HashMap<>();

        for (TableField<R, ?> field : tableFields) {
            JsonNode rootNode = objectMapper.createObjectNode();
            ImmutableCollection<JsonWriter<? super M>> jsonWriters = writerMap.get(field);
            for (JsonWriter<? super M> jsonWriter : jsonWriters) {
                String jsonPath = jsonWriter.getJsonPath();
                JsonNode jsonNode = jsonWriter.write(model);
                // Skip null values
                if (jsonNode != null && !jsonNode.isNull()) {
                    ((ObjectNode) rootNode).set(jsonPath, jsonNode);
                }
            }
            try {
                map.put(field, objectMapper.writeValueAsString(rootNode));
            } catch (JsonProcessingException e) {
                throw new RuntimeException();
            }
        }
        return map;
    }
}
