package ru.yandex.direct.jooqmapper.jsonwrite;

import java.util.List;

import com.google.common.collect.ImmutableMultimap;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.write.WriterBuilders;
import ru.yandex.direct.model.Model;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Билдер для создания {@link JooqJsonWriter}. Используйте вместе с {@link WriterBuilders}.
 */
public final class JooqJsonWriterBuilder<M extends Model> {

    private final ImmutableMultimap.Builder<TableField<? extends Record, ?>, JsonWriter<? super M>> writersBuilder =
            ImmutableMultimap.builder();

    private boolean isEmpty;

    private JooqJsonWriterBuilder() {
        isEmpty = true;
    }

    public static <M extends Model> JooqJsonWriterBuilder<M> builder() {
        return new JooqJsonWriterBuilder<>();
    }

    public static <M extends Model> JooqJsonWriterBuilder<M> builder(
            List<? extends JooqJsonWriter<? super M>> sourceMappers
    ) {
        JooqJsonWriterBuilder<M> builder = new JooqJsonWriterBuilder<>();
        var writers = mapList(sourceMappers, JooqJsonWriterBuilder::getWriters);

        writers.forEach(builder.writersBuilder::putAll);

        if (writers.stream().anyMatch(x -> !x.isEmpty())) {
            builder.setNotEmpty();
        }
        return builder;
    }

    private static <M extends Model> ImmutableMultimap<TableField<? extends Record, ?>, JsonWriter<? super M>> getWriters(
            JooqJsonWriter<M> sourceMapper) {
        return sourceMapper == null ? ImmutableMultimap.of() : sourceMapper.getWriters();
    }

    public <X extends Record, T> JooqJsonWriterBuilder<M> writeField(
            TableField<X, String> tableField,
            JsonWriter1Builder<? super M, T> writer1Builder
    ) {
        return putWriter(tableField, writer1Builder.build());
    }

    private JooqJsonWriterBuilder<M> putWriter(TableField<?, String> tableField, JsonWriter<? super M> writer) {
        setNotEmpty();
        writersBuilder.put(tableField, writer);
        return this;
    }

    private void setNotEmpty() {
        isEmpty = false;
    }

    public JooqJsonWriter<M> build() {
        return new JooqJsonWriter<>(writersBuilder.build());
    }

    public boolean isEmpty() {
        return isEmpty;
    }
}
