package ru.yandex.direct.jooqmapper.jsonwrite;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.jooqmapper.write.DefaultWriter;
import ru.yandex.direct.jooqmapper.write.PropertyValues;
import ru.yandex.direct.jooqmapper.write.WriterFunction1;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Objects.requireNonNull;

/**
 * Простая реализация интерфейса {@link JsonWriter}, которая вычисляет
 * записываемое в базу значение из одного свойства модели ({@link ModelProperty}).
 *
 * @param <M> тип модели.
 * @param <T> тип свойства модели.
 * @see DefaultWriter
 */
@ParametersAreNonnullByDefault
public class JsonWriter1<M extends Model, T> implements JsonWriter<M> {

    private final DefaultJsonWriter<M> innerWriter;
    private final String path;

    JsonWriter1(ModelProperty<? super M, T> property,
                String jsonPath,
                WriterFunction1<T, JsonNode> writerFunction) {
        requireNonNull(property, "model property is required");
        requireNonNull(jsonPath, "json path is required");
        requireNonNull(writerFunction, "writer function is required");

        Function<PropertyValues<M>, JsonNode> innerWriterFunction =
                propertyValues -> writerFunction.write(propertyValues.get(property));

        Set<ModelProperty<? super M, ?>> properties = ImmutableSet.of(property);
        this.innerWriter = new DefaultJsonWriter<>(properties, jsonPath, innerWriterFunction);
        this.path = jsonPath;
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getProperties() {
        return innerWriter.getProperties();
    }

    @Override
    public JsonNode write(M model) {
        return innerWriter.write(model);
    }

    @Override
    public String getJsonPath() {
        return path;
    }
}
