package ru.yandex.direct.jooqmapper.jsonwrite;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.function.Function;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

/**
 * Для построения {@link JooqWriter} используйте {@link JooqWriterBuilder},
 * а для построения {@link JooqMapper} - {@link JooqMapperBuilder}.
 */
public final class JsonWriterBuilders {

    public static final DateTimeFormatter LOCAL_DATE_TIME_FORMATTER =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private JsonWriterBuilders() {
    }

    public static JsonNode emumToJsonNode(Enum<?> v) {
        return v == null ? null : JsonNodeFactory.instance.textNode(v.name());
    }

    public static <T extends Enum<?>> JsonNode emumToJsonNode(Function<T, String> converter, T v) {
        return v == null ? null : JsonNodeFactory.instance.textNode(converter.apply(v));
    }


    public static JsonNode booleanToJsonNode(Boolean v) {
        return v == null ? null : JsonNodeFactory.instance.booleanNode(v);
    }

    public static JsonNode longToJsonNode(Long v) {
        return v == null ? null : JsonNodeFactory.instance.numberNode(v);
    }

    public static JsonNode integerToJsonNode(Integer v) {
        return v == null ? null : JsonNodeFactory.instance.numberNode(v);
    }

    public static JsonNode stringToJsonNode(String v) {
        return v == null ? null : JsonNodeFactory.instance.textNode(v);
    }

    public static JsonNode booleanToLongJsonNode(Boolean v) {
        return v == null ? null : JsonNodeFactory.instance.numberNode(v ? 1L : 0L);
    }

    public static JsonNode localDateTimeToStringJsonNode(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }

        return JsonNodeFactory.instance.textNode(localDateTime.format(LOCAL_DATE_TIME_FORMATTER));
    }

    public static JsonNode localDateToStringJsonNode(LocalDate localDate) {
        if (localDate == null) {
            return null;
        }

        return JsonNodeFactory.instance.textNode(localDate.format(DateTimeFormatter.ISO_LOCAL_DATE));
    }

    public static JsonNode bigDecimalToStringJsonNode(BigDecimal bigDecimal) {
        return bigDecimal == null ? null : JsonNodeFactory.instance.textNode(bigDecimal.toString());
    }

    public static <M extends Model, T extends Enum<?>> JsonWriter1Builder<? super M, T> fromEnumProperty(
            ModelProperty<? super M, T> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath).by(JsonWriterBuilders::emumToJsonNode);
    }

    public static <M extends Model, T extends Enum<?>> JsonWriter1Builder<? super M, T> fromConvertableEnumProperty(
            ModelProperty<? super M, T> property,
            Function<T, String> converter,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath)
                .by(enumValue -> JsonWriterBuilders.emumToJsonNode(converter, enumValue));
    }

    public static <M extends Model, T> JsonWriter1Builder.JsonWriter1WithPropertyStep<M, T> fromProperty(
            ModelProperty<? super M, T> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath);
    }

    public static <M extends Model> JsonWriter1Builder<? super M, Boolean> fromBooleanProperty(
            ModelProperty<? super M, Boolean> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath).by(JsonWriterBuilders::booleanToJsonNode);
    }

    public static <M extends Model> JsonWriter1Builder<? super M, Long> fromLongProperty(
            ModelProperty<? super M, Long> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath).by(JsonWriterBuilders::longToJsonNode);
    }

    public static <M extends Model> JsonWriter1Builder<? super M, Integer> fromIntegerProperty(
            ModelProperty<? super M, Integer> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath).by(JsonWriterBuilders::integerToJsonNode);
    }

    public static <M extends Model> JsonWriter1Builder<? super M, String> fromStringProperty(
            ModelProperty<? super M, String> property,
            String jsonPath) {
        return JsonWriter1Builder.fromProperty(property, jsonPath).by(JsonWriterBuilders::stringToJsonNode);
    }

}
