package ru.yandex.direct.jooqmapper.read;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.jooq.Field;
import org.jooq.Record;

import static java.util.Objects.requireNonNull;

/**
 * Читальщик значения свойства модели из {@link Record}.
 * Умеет отдавать требуемый список полей БД, которые должны присутствовать в {@link Record}.
 *
 * @param <T> тип читаемого значения.
 */
@ParametersAreNonnullByDefault
public class DefaultReader<T> implements Reader<T> {

    private final ImmutableSet<Field<?>> fields;
    private final Function<FieldValues, T> readerFunction;

    DefaultReader(Set<Field<?>> fields,
                  Function<FieldValues, T> readerFunction) {
        this.fields = ImmutableSet.copyOf(requireNonNull(fields, "fields set is required"));
        this.readerFunction = requireNonNull(readerFunction, "reader function is required");
    }

    @Override
    public Set<Field<?>> getRequiredDatabaseFields() {
        return fields;
    }

    @Override
    public T read(Record record) {
        FieldValues fieldValues = new FieldValues(fields, record);
        return readerFunction.apply(fieldValues);
    }
}
