package ru.yandex.direct.jooqmapper.read;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Objects.requireNonNull;

/**
 * Умеет читать из произвольных {@link Field}, в том числе вычисляемых.
 * Умеет по списку свойств модели отдавать список полей БД,
 * которые необходимо выбрать для их заполнения (см. {@link #getFieldsToRead(Collection)}).
 * <p>
 * Для создания используйте {@link JooqReaderBuilder}.
 *
 * @param <M> тип модели.
 */
@ParametersAreNonnullByDefault
public class JooqReader<M extends Model> {

    private final ImmutableMap<ModelProperty<? super M, ?>, Reader<?>> readers;
    // поля, которые нужно читать через агрегирующую функцию first в Yt
    private final Set<Field<?>> fieldsForFirst;

    public JooqReader(Map<ModelProperty<? super M, ?>, Reader<?>> readers) {
        this(readers, Collections.emptySet());
    }

    public JooqReader(Map<ModelProperty<? super M, ?>, Reader<?>> readers,
                      Set<Field<?>> fieldsForFirst) {
        this.readers = ImmutableMap.copyOf(requireNonNull(readers, "readers map is required"));
        this.fieldsForFirst = Set.copyOf(requireNonNull(fieldsForFirst, "fieldsForFirst set is required"));
        checkArgument(!readers.isEmpty(), "readers map must not be empty");
    }

    public ImmutableMap<ModelProperty<? super M, ?>, Reader<?>> getReaders() {
        return readers;
    }

    public ImmutableSet<ModelProperty<? super M, ?>> getReadableModelProperties() {
        return readers.keySet();
    }

    /**
     * Заполнение переданного экземпляра модели прочитанными из БД данными в виде {@link Record}.
     * <p>
     * Заполняет только те поля модели, для которых достаточно данных
     * в переданном экземпляре {@link Record}.
     * <p>
     * Если ни одно поле модели не было прочитано, генерируется исключение {@link IllegalStateException}.
     *
     * @param record jooq-овый результат чтения из БД.
     * @param model  экземпляр модели для заполнения.
     * @return переданный на вход экземпляр модели, заполненный данными из базы.
     */
    public M fromDb(Record record, M model) {
        return fromDb(record, model, ImmutableSet.copyOf(record.fields()));
    }

    public M fromDb(Record record, M model, Set<Field> availableFields) {
        boolean someDataWasRead = EntryStream.of(readers)
                .mapToInt(propAndReader -> readData(record, model, availableFields, propAndReader) ? 1 : 0)
                .sum() > 0;

        checkState(someDataWasRead, "there is not enough data in record to read at least one model property");

        return model;
    }

    private boolean readData(Record record, M model,
                             Set<Field> availableFields,
                             Map.Entry<ModelProperty<? super M, ?>, Reader<?>> propAndReader) {
        ModelProperty modelProperty = propAndReader.getKey();
        Reader reader = propAndReader.getValue();

        if (availableFields.containsAll(reader.getRequiredDatabaseFields())) {
            //noinspection unchecked
            modelProperty.set(model, reader.read(record));
            return true;
        }
        return false;
    }

    /**
     * Заполняет в переданном экземпляре модели указанные свойства.
     * Метод гарантирует, что все указанные свойства будут прочитаны и заполнены,
     * а если это невозможно по причине нехватки данных в {@link Record}
     * или отсутствии зарегистрированного экземпляра {@link Reader} для
     * чтения одного из указанных свойств, то будет сгенерировано исключение.
     * Эта гарантия НЕ означает, что прочитанное свойство модели будет обязательно
     * иметь значение, отличное от {@code null}, это зависит от логики чтения
     * и исходных данных.
     * <p>
     * В экземпляре {@link Record} должны присутствовать данные для заполнения
     * всех указанных свойств модели.
     * <p>
     * Если хотя бы одно из указанных свойств модели невозможно заполнить по причине нехватки данных,
     * будет сгенерировано исключение {@link IllegalArgumentException}.
     * <p>
     * Если хотя бы для одного из указанных свойств модели не зарегистрирован экземпляр {@link Reader},
     * будет сгенерировано исключение {@link IllegalArgumentException}.
     *
     * @param record          jooq-овый результат чтения из БД.
     * @param model           экземпляр модели для заполнения указанных свойств.
     * @param modelProperties список свойств модели, которые будут гарантированно прочитаны.
     * @return переданный на вход экземпляр модели с заполненными указанными свойствами.
     */
    public M fromDb(Record record, M model, List<ModelProperty<? super M, ?>> modelProperties) {
        checkArgument(!modelProperties.isEmpty(), "model properties to read must be provided");

        Set<Field> availableFields = ImmutableSet.copyOf(record.fields());

        for (ModelProperty modelProperty : modelProperties) {
            Reader reader = readers.get(modelProperty);
            checkArgument(reader != null,
                    "there are no registered readers for model property " + modelProperty);
            checkArgument(availableFields.containsAll(reader.getRequiredDatabaseFields()),
                    "there is not enough data in record to read model property " + modelProperty);
            modelProperty.set(model, reader.read(record));
        }
        return model;
    }

    /**
     * @param modelProperties список свойств модели.
     * @return список полей БД, которые необходимо прочитать для заполнения указанных свойств модели.
     */
    public Set<Field<?>> getFieldsToRead(Collection<ModelProperty<?, ?>> modelProperties) {
        return StreamEx.of(modelProperties)
                .flatCollection(modelProperty -> {
                    Reader<?> reader = readers.get(modelProperty);
                    checkArgument(reader != null,
                            "there are no registered readers for model property " + modelProperty);
                    return reader.getRequiredDatabaseFields();
                })
                .toSet();
    }

    /**
     * @return список полей БД, которые необходимо прочитать для заполнения
     * всех свойств модели, которые умеет читать данный маппер.
     */
    public Set<Field<?>> getFieldsToRead() {
        return StreamEx.of(readers.values())
                .flatCollection(Reader::getRequiredDatabaseFields)
                .remove(fieldsForFirst::contains)
                .toSet();
    }

    /**
     * @return список полей БД, которые необходимо прочитать через агрегирующую функцию first, для заполнения
     * всех свойств модели, которые умеет читать данный маппер.
     */
    public Set<Field<?>> getFieldsForFirstToRead() {
        return fieldsForFirst;
    }

    /**
     * @see #canReadAtLeastOneProperty(Collection)
     */
    public boolean canReadAtLeastOneProperty(Field<?>[] fieldsCandidates) {
        Set<? super Field<?>> fieldsCandidatesSet = ImmutableSet.copyOf(fieldsCandidates);
        return canReadAtLeastOneProperty(fieldsCandidatesSet);
    }

    /**
     * Возвращает {@code true}, если существует возможность прочитать хотя бы
     * одно зарегистрированное свойство модели из указанных полей базы,
     * в противном случае {@code false}.
     *
     * Например, если для чтения определенного зарегистрированного свойства {@link ModelProperty}
     * требуется наличие значения двух полей базы в записи {@link Record}, и при этом в переданной
     * коллекции присутствует только одно из них, то метод будет считать, что данное свойство
     * не может быть прочитано. Если же будут присутствовать оба, значит, для чтения свойства
     * будет достаточно данных.
     *
     * Логика метода согласована с методами {@link #fromDb(Record, Model)} и
     * {@link #fromDb(Record, Model, List)}.
     */
    public boolean canReadAtLeastOneProperty(Collection<? super Field<?>> fieldsCandidates) {
        Set<? super Field<?>> fieldsCandidatesSet = ImmutableSet.copyOf(fieldsCandidates);
        return StreamEx.of(readers.values())
                .map(Reader::getRequiredDatabaseFields)
                .anyMatch(fieldsCandidatesSet::containsAll);
    }
}
