package ru.yandex.direct.jooqmapper.read;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;

/**
 * Билдер для создания {@link JooqReader}. Используйте вместе с {@link ReaderBuilders}.
 */
public class JooqReaderBuilder<M extends Model> {

    protected final Map<ModelProperty<? super M, ?>, Reader<?>> readers = new HashMap<>();
    // поля, которые нужно читать через агрегирующую функцию first в Yt
    protected final Set<Field<?>> fieldsForFirst = new HashSet<>();

    protected JooqReaderBuilder() {
    }

    public static <M extends Model> JooqReaderBuilder<M> builder() {
        return new JooqReaderBuilder<>();
    }

    public static <M extends Model> JooqReaderBuilder<M> builder(
            JooqReader<? super M> sourceMapper) {
        JooqReaderBuilder<M> builder = new JooqReaderBuilder<>();
        builder.readers.putAll(sourceMapper.getReaders());
        builder.fieldsForFirst.addAll(sourceMapper.getFieldsForFirstToRead());
        return builder;
    }

    public static <M extends Model> JooqReaderBuilder<M> builder(
            List<? extends JooqReader<? super M>> sourceMappers) {
        JooqReaderBuilder<M> builder = new JooqReaderBuilder<>();
        sourceMappers.forEach(mapper -> builder.readers.putAll(getReaders(mapper)));
        sourceMappers.forEach(mapper -> builder.fieldsForFirst.addAll(getFieldsForFirst(mapper)));
        return builder;
    }

    private static <M extends Model> Map<? extends ModelProperty<? super M, ?>, ? extends Reader<?>> getReaders(
            @Nullable JooqReader<? super M> sourceMapper) {
        return sourceMapper == null ?
                Map.of() :
                sourceMapper.getReaders();
    }

    private static Set<Field<?>> getFieldsForFirst(@Nullable JooqReader<?> sourceMapper) {
        return sourceMapper == null ?
                Set.of() :
                sourceMapper.getFieldsForFirstToRead();
    }

    public <T extends Model> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                               JooqReaderWithSupplier<T> reader) {
        putReader(modelProperty, new Reader<T>() {
            @Override
            public Set<Field<?>> getRequiredDatabaseFields() {
                return flatMapToSet(reader.getReaders().values(), Reader::getRequiredDatabaseFields);
            }

            @Override
            public T read(Record record) {
                return reader.fromDb(record);
            }
        });
        fieldsForFirst.addAll(reader.getFieldsForFirstToRead());
        return this;
    }

    public <T, R> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                    Reader1Builder<R, T> reader1Builder) {
        return putReader(modelProperty, reader1Builder.build());
    }

    public <T> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                 Reader1Builder.Reader1WithFieldStep<T> reader1Builder) {
        return putReader(modelProperty, reader1Builder.by(t -> t).build());
    }

    public <T, R1, R2> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                         Reader2Builder<R1, R2, T> reader2Builder) {
        return putReader(modelProperty, reader2Builder.build());
    }

    public <T, R1, R2, R3> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                             Reader3Builder<R1, R2, R3, T> reader3Builder) {
        return putReader(modelProperty, reader3Builder.build());
    }

    public <T> JooqReaderBuilder<M> readProperty(ModelProperty<? super M, T> modelProperty,
                                                 DefaultReaderBuilder<T> readerBuilder) {
        return putReader(modelProperty, readerBuilder.build());
    }

    public <T, R> JooqReaderBuilder<M> readPropertyForFirst(ModelProperty<? super M, T> modelProperty,
                                                            Reader1Builder<R, T> reader1Builder) {
        return putReaderForFirst(modelProperty, reader1Builder.build());
    }

    public <T> JooqReaderBuilder<M> readPropertyForFirst(ModelProperty<? super M, T> modelProperty,
                                                         Reader1Builder.Reader1WithFieldStep<T> reader1Builder) {
        return putReaderForFirst(modelProperty, reader1Builder.by(t -> t).build());
    }

    public <T, R1, R2> JooqReaderBuilder<M> readPropertyForFirst(ModelProperty<? super M, T> modelProperty,
                                                                 Reader2Builder<R1, R2, T> reader2Builder) {
        return putReaderForFirst(modelProperty, reader2Builder.build());
    }

    public <T, R1, R2, R3> JooqReaderBuilder<M> readPropertyForFirst(ModelProperty<? super M, T> modelProperty,
                                                                     Reader3Builder<R1, R2, R3, T> reader3Builder) {
        return putReaderForFirst(modelProperty, reader3Builder.build());
    }

    public <T> JooqReaderBuilder<M> readPropertyForFirst(ModelProperty<? super M, T> modelProperty,
                                                         DefaultReaderBuilder<T> readerBuilder) {
        return putReaderForFirst(modelProperty, readerBuilder.build());
    }

    private <T> JooqReaderBuilder<M> putReader(ModelProperty<? super M, T> modelProperty, Reader<T> reader) {
        checkArgument(!readers.containsKey(modelProperty),
                "attempt to register more than one reader for property %s", modelProperty);
        readers.put(modelProperty, reader);
        return this;
    }

    /**
     * Добавление ридера для чтения поля, получаемого через агрегирующую функцию first в Yt
     */
    private <T> JooqReaderBuilder<M> putReaderForFirst(ModelProperty<? super M, T> modelProperty, Reader<T> reader) {
        putReader(modelProperty, reader);
        fieldsForFirst.addAll(reader.getRequiredDatabaseFields());
        return this;
    }

    public JooqReader<M> build() {
        return new JooqReader<>(readers, fieldsForFirst);
    }

    public boolean isEmpty() {
        return readers.isEmpty();
    }
}
