package ru.yandex.direct.jooqmapper.read;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Objects.requireNonNull;

/**
 * Умеет читать из произвольных {@link Field}, в том числе вычисляемых.
 * Умеет по списку свойств модели отдавать список полей БД,
 * которые необходимо выбрать для их заполнения (см. {@link #getFieldsToRead(Collection)}.
 * Умеет при чтении данных из {@link Record} получать экземпляр модели для заполнения
 * с помощью переданного {@code Supplier<M> modelSupplier}.
 * <p>
 * Для создания используйте {@link JooqReaderWithSupplierBuilder}.
 *
 * @param <M> тип модели.
 */
@ParametersAreNonnullByDefault
public class JooqReaderWithSupplier<M extends Model> extends JooqReader<M> {

    private final Supplier<M> modelSupplier;

    public JooqReaderWithSupplier(Map<ModelProperty<? super M, ?>, Reader<?>> readers,
                                  Supplier<M> modelSupplier) {
        super(readers);
        this.modelSupplier = requireNonNull(modelSupplier, "model supplier is required");
    }

    /**
     * @param fieldsForFirst - поля, которые нужно читать через агрегирующую функцию first в Yt
     */
    public JooqReaderWithSupplier(Map<ModelProperty<? super M, ?>, Reader<?>> readers,
                                  Set<Field<?>> fieldsForFirst,
                                  Supplier<M> modelSupplier) {
        super(readers, fieldsForFirst);
        this.modelSupplier = requireNonNull(modelSupplier, "model supplier is required");
    }

    /**
     * Заполнение экземпляра модели прочитанными из БД данными в виде {@link Record}.
     * Экземпляр модели получается с помощью переданного при инициализации
     * {@code Supplier<M> modelSupplier}.
     * <p>
     * Заполняет только те поля модели, для которых достаточно данных
     * в переданном экземпляре {@link Record}.
     * <p>
     * Если ни одно поле модели не было прочитано, генерируется исключение {@link IllegalStateException}.
     *
     * @param record jooq-овый результат чтения из БД.
     * @return переданный на вход экземпляр модели, заполненный данными из базы.
     */
    public M fromDb(Record record) {
        return fromDb(record, modelSupplier.get());
    }

    /**
     * Заполняет в экземпляре модели указанные свойства. Экземпляр модели получается
     * с помощью переданного при инициализации {@code Supplier<M> modelSupplier}.
     * Метод гарантирует, что все указанные свойства будут прочитаны и заполнены,
     * а если это невозможно по причине нехватки данных в {@link Record}
     * или отсутствии зарегистрированного экземпляра {@link Reader} для
     * чтения одного из указанных свойств, то будет сгенерировано исключение.
     * Эта гарантия НЕ означает, что прочитанное свойство модели будет обязательно
     * иметь значение, отличное от {@code null}, это зависит от логики чтения
     * и исходных данных.
     * <p>
     * В экземпляре {@link Record} должны присутствовать данные для заполнения
     * всех указанных свойств модели.
     * <p>
     * Если хотя бы одно из указанных свойств модели невозможно заполнить по причине нехватки данных,
     * будет сгенерировано исключение {@link IllegalArgumentException}.
     * <p>
     * Если хотя бы для одного из указанных свойств модели не зарегистрирован экземпляр {@link Reader},
     * будет сгенерировано исключение {@link IllegalArgumentException}.
     *
     * @param record          jooq-овый результат чтения из БД.
     * @param modelProperties список свойств модели, которые будут гарантированно прочитаны.
     * @return переданный на вход экземпляр модели с заполненными указанными свойствами.
     */
    public M fromDb(Record record, List<ModelProperty<? super M, ?>> modelProperties) {
        return fromDb(record, modelSupplier.get(), modelProperties);
    }
}
