package ru.yandex.direct.jooqmapper.read;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.jooq.Field;
import org.jooq.Record;

import static java.util.Objects.requireNonNull;

/**
 * Простая реализация интерфейса {@link Reader}, читающая значение
 * свойства модели из двух полей ({@link Field}).
 *
 * @param <R1> тип первого читаемого из базы значения.
 * @param <R2> тип второго читаемого из базы значения.
 * @param <T>  тип свойства модели, в которое осуществляется чтение из двух полей БД.
 * @see DefaultReader
 */
@ParametersAreNonnullByDefault
public class Reader2<R1, R2, T> implements Reader<T> {

    private final DefaultReader<T> innerReader;

    Reader2(Field<R1> field1, Field<R2> field2,
            ReaderFunction2<R1, R2, T> readerFunction) {
        requireNonNull(field1, "first field is required");
        requireNonNull(field2, "second field is required");
        requireNonNull(readerFunction, "reader function is required");

        Function<FieldValues, T> innerReaderFunction =
                fieldValues -> readerFunction.read(fieldValues.getValue(field1), fieldValues.getValue(field2));

        Set<Field<?>> fields = ImmutableSet.of(field1, field2);
        this.innerReader = new DefaultReader<>(fields, innerReaderFunction);
    }

    @Override
    public final Set<Field<?>> getRequiredDatabaseFields() {
        return innerReader.getRequiredDatabaseFields();
    }

    @Override
    public final T read(Record record) {
        return innerReader.read(record);
    }
}
