package ru.yandex.direct.jooqmapper.write;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Вычисляет значение для записи в таблицу БД из набора свойств модели.
 * Умеет отдавать свойства модели, которые будут прочитаны для записи значения в БД.
 *
 * @param <M> тип модели.
 * @param <R> тип записываемого значения.
 */
@ParametersAreNonnullByDefault
public class DefaultWriter<M extends Model, R> implements Writer<M, R> {

    private final ImmutableSet<ModelProperty<? super M, ?>> properties;
    private final Function<PropertyValues<M>, R> writerFunction;

    DefaultWriter(Set<ModelProperty<? super M, ?>> properties,
                  Function<PropertyValues<M>, R> writerFunction) {
        this.properties = ImmutableSet.copyOf(requireNonNull(properties, "properties set is required"));
        this.writerFunction = requireNonNull(writerFunction, "writer function is required");
        checkArgument(!properties.isEmpty(), "properties set must not be empty");
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getProperties() {
        return properties;
    }

    @Override
    public R write(M model) {
        PropertyValues<M> propertyValues = new PropertyValues<>(properties, model);
        return writerFunction.apply(propertyValues);
    }
}
