package ru.yandex.direct.jooqmapper.write;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqModelToDbFieldValuesMapper;
import ru.yandex.direct.jooqmapper.JooqModelToDbMapper;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.utils.Collectors.nullFriendlyMapCollector;

/**
 * @see ru.yandex.direct.jooqmapper.kotlinwrite.KotlinJooqWriter
 * менять совместно
 */
@ParametersAreNonnullByDefault
public class JooqWriter<M extends Model> implements JooqModelToDbMapper<M>, JooqModelToDbFieldValuesMapper<M> {

    private final ImmutableMap<TableField<?, ?>, Writer<? super M, ?>> writers;
    private final Map<TableField<?, ?>, Writer<? super M, ?>> explicitWriters;

    private final ImmutableSet<ModelProperty<? super M, ?>> writableModelProperties;

    public JooqWriter(Map<TableField<?, ?>, Writer<? super M, ?>> writers,
                      Set<TableField<?, ?>> explicitFields) {
        this.writers = ImmutableMap.copyOf(requireNonNull(writers, "writers map is required"));
        this.explicitWriters = EntryStream.of(writers)
                .filterKeys(explicitFields::contains)
                .toMap();

        // ModelProperty, которые обрабатывют writer'ы
        this.writableModelProperties = ImmutableSet.copyOf(StreamEx.ofValues(writers)
                .map(Writer::getProperties)
                .flatMap(Collection::stream)
                .toSet());

        checkArgument(!writers.isEmpty(), "writers map must not be empty");

        ImmutableMultimap.Builder<ModelProperty<? super M, ?>, Writer<? super M, ?>> propertyToWritersMapBuilder =
                ImmutableMultimap.builder();
        for (Writer<? super M, ?> writer : writers.values()) {
            for (ModelProperty<? super M, ?> modelProperty : writer.getProperties()) {
                propertyToWritersMapBuilder.put(modelProperty, writer);
            }
        }
    }

    ImmutableMap<TableField<?, ?>, Writer<? super M, ?>> getWriters() {
        return writers;
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(M model, Table<R> table) {
        // noinspection unchecked приведение типа безопасно, т.к. предварительно делается filter.
        return EntryStream.of(writers)
                .filterKeys(tableField -> tableField.getTable().equals(table))
                .mapKeys(tableField -> (TableField<R, ?>) tableField)
                .mapValues(writer -> writer.write(model))
                // стандартный коллектор toMap не выносит null значений (кидает NPE)
                .collect(nullFriendlyMapCollector());
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(
            M model,
            Table<R> table,
            Set<ModelProperty<? super M, ?>> requestedProperties,
            Set<ModelProperty<? super M, ?>> explicitValueProperties) {
        var values = getDbFieldValuesInternal(writers, model,
                requestedProperties, table);
        var explicitValues = getDbFieldValuesInternal(explicitWriters, model,
                explicitValueProperties, table);
        ((Map) values).putAll(explicitValues);
        return values;
    }

    private <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValuesInternal(
            Map<TableField<?, ?>, Writer<? super M, ?>> writersToUse,
            M model,
            Set<ModelProperty<? super M, ?>> properties,
            Table<R> table) {
        // noinspection unchecked приведение типа безопасно, т.к. предварительно делается filter.
        return EntryStream.of(writersToUse)
                .filterKeys(tableField -> tableField.getTable().equals(table))
                .mapKeys(tableField -> (TableField<R, ?>) tableField)
                .filterValues(writer -> !Sets.intersection(properties, writer.getProperties()).isEmpty())
                .mapValues(writer -> writer.write(model))
                // стандартный коллектор toMap не выносит null значений (кидает NPE)
                .collect(nullFriendlyMapCollector());
    }

    public ImmutableSet<ModelProperty<? super M, ?>> getWritableModelProperties() {
        return writableModelProperties;
    }
}
