package ru.yandex.direct.jooqmapper.write;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.model.Model;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Билдер для создания {@link JooqWriter}. Используйте вместе с {@link WriterBuilders}.
 */
public final class JooqWriterBuilder<M extends Model> {

    private final Map<TableField<?, ?>, Writer<? super M, ?>> writers = new HashMap<>();
    private final Set<TableField<?, ?>> explicitFields = new HashSet<>();

    private JooqWriterBuilder() {
    }

    public static <M extends Model> JooqWriterBuilder<M> builder() {
        return new JooqWriterBuilder<>();
    }

    public static <M extends Model> JooqWriterBuilder<M> builder(JooqWriter<? super M> sourceMapper) {
        JooqWriterBuilder<M> builder = new JooqWriterBuilder<>();
        builder.writers.putAll(sourceMapper.getWriters());
        return builder;
    }

    public static <M extends Model> JooqWriterBuilder<M> builder(
            List<? extends JooqWriter<? super M>> sourceMappers) {
        JooqWriterBuilder<M> builder = new JooqWriterBuilder<>();
        sourceMappers.forEach(mapper -> builder.writers.putAll(getWriters(mapper)));
        return builder;
    }

    private static <M extends Model> Map<? extends TableField<?, ?>, ? extends Writer<? super M, ?>> getWriters(
            @Nullable JooqWriter<? super M> sourceMapper) {
        return sourceMapper == null ? Map.of() : sourceMapper.getWriters();
    }

    public <X extends Record, T, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer1Builder<? super M, T, R> writer1Builder
    ) {
        return putWriter(tableField, writer1Builder.build());
    }

    public <X extends Record, T, R> JooqWriterBuilder<M> writeFieldExplicitly(
            TableField<X, R> tableField,
            Writer1Builder<? super M, T, R> writer1Builder
    ) {
        explicitFields.add(tableField);
        return writeField(tableField, writer1Builder);
    }

    public <X extends Record, T, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            WriterToFieldBuilder<? super M, T, R> writerToFieldBuilder
    ) {
        return putWriterField(tableField, writerToFieldBuilder.build());
    }

    public <T> JooqWriterBuilder<M> writeField(
            TableField<?, T> tableField,
            Writer1Builder.Writer1WithPropertyStep<M, T> writer1Builder
    ) {
        return putWriter(tableField, writer1Builder.by(t -> t).build());
    }

    public <X extends Record, T1, T2, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer2Builder<? super M, T1, T2,
                    R> writer2Builder
    ) {
        return putWriter(tableField, writer2Builder.build());
    }

    public <X extends Record, T1, T2, T3, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer3Builder<? super M, T1,
                    T2, T3, R> writer3Builder
    ) {
        return putWriter(tableField, writer3Builder.build());
    }

    public <X extends Record, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            DefaultWriterBuilder<? super M, R> writer3Builder
    ) {
        return putWriter(tableField, writer3Builder.build());
    }

    public <X extends Record, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            SupplierWriterBuilder<? super M, R> supplierWriterBuilder
    ) {
        return putWriter(tableField, supplierWriterBuilder.build());
    }

    public <X extends Record, R> JooqWriterBuilder<M> writeField(
            TableField<X, R> tableField,
            FunctionWriterBuilder<? super M, R> functionWriterBuilder
    ) {
        return putWriter(tableField, functionWriterBuilder.build());
    }

    private <T> JooqWriterBuilder<M> putWriter(TableField<?, T> tableField, Writer<? super M, T> writer) {
        checkArgument(!writers.containsKey(tableField),
                "attempt to register more than one writer for field %s", tableField);
        writers.put(tableField, writer);
        return this;
    }

    private <T> JooqWriterBuilder<M> putWriterField(TableField<?, T> tableField,
                                                    Writer<? super M, Field<T>> writer) {
        checkArgument(!writers.containsKey(tableField),
                "attempt to register more than one writer for field %s", tableField);
        writers.put(tableField, writer);
        return this;
    }

    public JooqWriter<M> build() {
        return new JooqWriter<>(writers, explicitFields);
    }

    public boolean isEmpty() {
        return writers.isEmpty();
    }
}
