package ru.yandex.direct.jooqmapper.write;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Objects.requireNonNull;

/**
 * Простая реализация интерфейса {@link Writer}, которая вычисляет
 * записываемое в базу значение из двух свойств модели ({@link ModelProperty}).
 *
 * @param <M>  тип модели.
 * @param <T1> тип первого свойства модели.
 * @param <T2> тип второго свойства модели.
 * @param <R>  тип записываемого в базу значения.
 * @see DefaultWriter
 */
@ParametersAreNonnullByDefault
public class Writer2<M extends Model, T1, T2, R> implements Writer<M, R> {

    private final DefaultWriter<M, R> innerWriter;

    Writer2(ModelProperty<? super M, T1> property1, ModelProperty<? super M, T2> property2,
            WriterFunction2<T1, T2, R> writerFunction) {
        requireNonNull(property1, "first model property is required");
        requireNonNull(property2, "second model property is required");
        requireNonNull(writerFunction, "writer function is required");

        Function<PropertyValues<M>, R> innerWriterFunction =
                propertyValues -> writerFunction.write(propertyValues.get(property1),
                        propertyValues.get(property2));

        Set<ModelProperty<? super M, ?>> properties =
                ImmutableSet.<ModelProperty<? super M, ?>>of(property1, property2);
        innerWriter = new DefaultWriter<>(properties, innerWriterFunction);
    }

    @Override
    public Set<ModelProperty<? super M, ?>> getProperties() {
        return innerWriter.getProperties();
    }

    @Override
    public R write(M model) {
        return innerWriter.write(model);
    }
}
