package ru.yandex.direct.jooqmapperhelper;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertQuery;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqModelToDbMapper;
import ru.yandex.direct.model.Model;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

/**
 * Хэлпер для удобства выполнения insert'ов в базу с помощью {@link JooqMapper}.
 * С ним удобнее создавать одну запись в базе из нескольких объектов,
 * а так же создавать несколько записей одним запросом.
 * <p>
 * Пример использования:
 * <pre> {@code
 *
 *  public class Address {
 *      private String city;
 *  }
 *
 *  public class Person {
 *      private String name;
 *      private Address address;
 *  }
 *
 *  public class PersonRepository {
 *
 *      private OldJooqMapper<Person> personMapper;
 *      private OldJooqMapper<Address> addressMapper;
 *
 *
 *      public void addPersons(List<Person> persons) {
 *          if (persons.isEmpty()) {
 *              return;
 *          }
 *          InsertHelper<PersonsRecord> insertHelper
 *                  = new InsertHelper<>(dslContext(), PERSONS);
 *          for (Person person : persons) {
 *              insertHelper
 *                  .add(personMapper, person)
 *                  .add(addressMapper, person.getAddress())
 *                  .set(PERSONS.FIELD_THAT_DOES_NOT_PRESENT_IN_MODEL, calcThatField()),
 *                  .newRecord();
 *          }
 *          insertHelper.execute();
 *      }
 * }
 *
 * }</pre>
 *
 * @param <R>
 */
public class InsertHelper<R extends Record> {

    private final Table<R> table;
    private final InsertQuery<R> insertQuery;

    private boolean hasAddedRecords = false;
    private boolean onDuplicateKeyUpdate = false;

    public InsertHelper(DSLContext dslContext, Table<R> table) {
        this.table = table;
        this.insertQuery = dslContext.insertQuery(table);
    }

    /**
     * Сохранить несколько объектов модели в базу новыми записями.
     * <p>
     * Если объекты не переданы (пустой Iterable), ничего не делает.
     * <p>
     * Пример:
     * <code>
     * class Cat extends Model { ... }
     * <p>
     * class CatsRecord extends UpdatableRecordImpl&lt;CatsRecord&gt; { ... }
     * class Cats extends TableImpl&lt;CatsRecord&gt; {
     * public static final Cats CATS = new Cats();
     * ...
     * }
     * <p>
     * class CatsAuxiliaryInfoRecord extends UpdatableRecordImpl&lt;CatsAuxiliaryInfoRecord&gt; { ... }
     * class CatsAuxiliaryInfo extends TableImpl&lt;CatsAuxiliaryInfoRecord&gt; {
     * public static final Cats CATS_AUXILIARY_INFO = new CatsAuxiliaryInfo();
     * ...
     * }
     * <p>
     * class CatRepository {
     * JooqModelToDbMapper&lt;Cat&gt; mapper = new OldJooqMapperBuilder&lt;&gt;()
     * ...
     * .build();
     * <p>
     * public static void saveExampleCats(DSLContext dslContext) {
     * List&lt;Cat&gt; cats = Arrays.asList(new Cat().withField1(value1), new Cat().withField2(value2));
     * InsertHelper.saveModelObjectsToDbTable(dslContext, Cats.CATS, mapper, cats);
     * InsertHelper.saveModelObjectsToDbTable(dslContext, CatsAuxiliaryInfo.CATS_AUXILIARY_INFO, mapper, cats);
     * }
     * }
     * </code>
     *
     * @param dslContext соединение с базой
     * @param table      таблица, в которую надо писать, например, Phrases.PHRASES
     * @param mapper     любая реализация {@link JooqModelToDbMapper}, чтобы превратить объекты моделей в строки в базе
     * @param data       объекты моделей, которые надо сохранить
     * @param <R>        тип записи; обычно явно задавать не надо, потому что он участвует в определении
     *                   типа параметра table
     * @param <M>        тип класса модели; обычно явно задавать не надо, потому что он участвует в определении типа
     *                   параметра data
     */
    public static <R extends Record, M extends Model> void saveModelObjectsToDbTable(
            DSLContext dslContext,
            Table<R> table,
            JooqModelToDbMapper<M> mapper,
            Iterable<? extends M> data
    ) {
        if (!data.iterator().hasNext()) {
            return;
        }

        InsertHelper<R> helper = new InsertHelper<>(dslContext, table);
        helper.addAll(mapper, data);
        helper.execute();
    }

    /**
     * Для переданного списка моделей добавляет записи в БД и возвращает
     * список id добавленных колонок.
     */
    public static <R extends Record, M extends Model> List<Long> addModelsAndReturnIds(
            DSLContext dslContext,
            Table<R> table,
            JooqModelToDbMapper<M> mapper,
            List<? extends M> models,
            TableField<R, Long> tableIdField
    ) {
        var insertHelper = new InsertHelper<>(dslContext, table);
        insertHelper.addAll(mapper, models);
        return insertHelper.executeIfRecordsAddedAndReturn(tableIdField);
    }

    public <M> InsertHelper<R> add(JooqModelToDbMapper<M> jooqMapper, M model) {
        Map<TableField<R, ?>, ?> fieldValues = jooqMapper.getDbFieldValues(model, table);
        insertQuery.addValues(fieldValues);
        hasAddedRecords = true;
        return this;
    }

    public <T> InsertHelper<R> set(Field<T> field, T value) {
        if (onDuplicateKeyUpdate) {
            insertQuery.addValueForUpdate(field, value);
        } else {
            insertQuery.addValue(field, value);
        }
        hasAddedRecords = true;

        return this;
    }

    public <T> InsertHelper<R> set(Field<T> field, Field<T> value) {
        if (onDuplicateKeyUpdate) {
            insertQuery.addValueForUpdate(field, value);
        } else {
            insertQuery.addValue(field, value);
        }
        hasAddedRecords = true;
        return this;
    }

    /**
     * Метод добавления списка моделей в БД
     *
     * @param jooqMapper маппинг модели на БД
     * @param models     список сохраняемых моделей
     * @param <M>        сохраняемый тип объекта
     * @return возвращается сам хелпер для fluent интерфейса
     */
    public <M> InsertHelper<R> addAll(JooqModelToDbMapper<M> jooqMapper, Iterable<? extends M> models) {
        for (M model : models) {
            add(jooqMapper, model).newRecord();
        }
        return this;
    }

    /**
     * Метод добавления указанной модели при поомощи преобразования mappingModel из списка моделей в БД.
     * Добавляются модели прошедшие проверку @param addingCondition
     *
     * @param jooqSubmodelMapper маппинг подмодели на БД
     * @param models             список моделей, для которых нужно сохранить подмодель
     * @param <M>                тип модели
     * @param <S>                тип сохраняемой подмодели
     * @return возвращается сам хелпер для fluent интерфейса
     */
    public <M, S> InsertHelper<R> addAll(JooqModelToDbMapper<S> jooqSubmodelMapper, Iterable<M> models,
                                         Function<M, S> mappingModel,
                                         Predicate<M> addingCondition) {
        for (M model : models) {
            if (addingCondition.test(model)) {
                S mappedModel = mappingModel.apply(model);
                add(jooqSubmodelMapper, mappedModel).newRecord();
            }
        }
        return this;
    }


    public InsertHelper<R> newRecord() {
        checkState(hasAddedRecords, "add() or set() must be called at least once before newRecord()");
        insertQuery.newRecord();
        return this;
    }

    public InsertHelper<R> onDuplicateKeyUpdate() {
        checkState(hasAddedRecords, "add() or set() must be called at least once before onDuplicateKeyUpdate()");
        insertQuery.onDuplicateKeyUpdate(true);
        onDuplicateKeyUpdate = true;
        return this;
    }

    public InsertHelper<R> onDuplicateKeyIgnore() {
        if (hasAddedRecords) {
            insertQuery.onDuplicateKeyIgnore(true);
        }
        return this;
    }

    public int execute() {
        checkState(hasAddedRecords, "add() or set() must be called at least once before execute()");
        return insertQuery.execute();
    }

    public int executeIfRecordsAdded() {
        if (hasAddedRecords) {
            return insertQuery.execute();
        }

        return 0;
    }

    public <T> List<T> executeIfRecordsAddedAndReturn(Field<T> returningField) {
        if (hasAddedRecords) {
            insertQuery.setReturning(returningField);
            insertQuery.execute();
            return insertQuery
                    .getReturnedRecords().stream()
                    .map(r -> r.getValue(returningField))
                    .collect(toList());
        }

        return emptyList();
    }

    public boolean hasAddedRecords() {
        return hasAddedRecords;
    }

}
