package ru.yandex.direct.jooqmapperhelper;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

/**
 * Билдер для построения VALUES для jooq для обновления модели.
 * Для массовых апдейтов хочется делать запросы вида
 * <pre>
 *     UPDATE table
 *        SET name = case id when 1 then "x" when 5 then "y" else name end,
 *            description = case id when 6 then "desc" else description end
 *      WHERE id in (1,5,6)
 * </pre>
 * Билдер собирает словарь: поля -> CASE-выражения и множество id, по которым есть хотя бы одно изменение
 *
 * @param <R> - класс jooq-record-а
 * @param <M> - класс модели
 */
public class JooqUpdateBuilder<R extends Record, M extends ModelWithId> {
    private final TableField<R, Long> idField;
    private final Multimap<ModelProperty, M> propToModelObj;

    // результирующий map
    private final Map<TableField<R, ?>, Field<?>> values = new HashMap<>();
    private final Set<Long> changedIds = new HashSet<>();

    /**
     * @param idField jooq поле, в котором содержится id
     */
    public JooqUpdateBuilder(TableField<R, Long> idField,
                             Collection<? extends AppliedChanges<? extends M>> appliedChanges) {
        this.idField = idField;
        this.propToModelObj = getPropertyToModelMultimap(appliedChanges);
    }

    /**
     * Проверить, изменилось ли переданное свойство prop хотя бы у одного объекта.
     * Если поменялось - добавить в конструируемый объект соответсвующее поле и sql CASE(),
     * сопоставляющий id объекта новому значению
     *
     * @param prop        - проверяемое свойство
     * @param dataField   - jooq поле таблицы для обновления
     * @param transformer -
     * @param <F>         - тип значения свойства
     * @param <T>         - тип значения поля таблицы
     * @return - добавился update хотя бы для одного id
     */
    public <F, T> boolean processProperty(
            ModelProperty<? super M, F> prop,
            TableField<R, T> dataField,
            Function<F, T> transformer) {
        return processProperties(
                Collections.singleton(prop), dataField,
                (M model) -> transformer.apply(prop.get(model)));
    }

    /**
     * Проверить, изменились ли переданные свойства {@code complexProps} хотя бы у одного объекта.
     * Если поменялось - добавить в конструируемый update значение, полученное с помощью {@code modelTransformer}
     * из объекта с изменившимися свойствами, и sql CASE(), сопоставляющий id объекта новому значению
     *
     * @param complexProps - проверяемые на изменение свойства
     * @param dataField    - jooq поле таблицы для обновления
     * @param <T>          - тип значения поля таблицы
     * @return - добавился update хотя бы для одного id
     */
    public <T> boolean processProperties(
            Collection<? extends ModelProperty<? super M, ?>> complexProps,
            TableField<R, T> dataField,
            Function<M, T> modelTransformer) {
        Map<Long, T> propValueMap = StreamEx.of(complexProps)
                .flatMap(prop -> propToModelObj.get(prop).stream())
                .distinct(ModelWithId::getId)
                // стандартный коллектор toMap не выносит null значений (кидает NPE)
                .collect(HashMap::new,
                        (m, model) -> m.put(model.getId(), modelTransformer.apply(model)),
                        HashMap::putAll);

        if (propValueMap.isEmpty()) {
            return false;
        }

        Field<T> choose = JooqMapperUtils.makeCaseStatement(idField, dataField, propValueMap);

        values.put(dataField, choose);
        changedIds.addAll(propValueMap.keySet());

        return true;
    }

    /**
     * {@see JooqUpdateBuilder.processProperty} с идентичным трансформером
     */
    public <F> boolean processProperty(ModelProperty<? super M, F> prop, TableField<R, F> dataField) {
        return processProperty(prop, dataField, Function.identity());
    }

    /**
     * @return - подготовленный словарь: поле -> sql
     */
    @Nonnull
    @SuppressWarnings("squid:S1452")
    public Map<TableField<R, ?>, Field<?>> getValues() {
        return values;
    }

    /**
     * @return - множество изменившихся id
     */
    @Nonnull
    public Set<Long> getChangedIds() {
        return changedIds;
    }

    /**
     * Ультимативно проставить значение
     */
    @SuppressWarnings("unused")
    public <V> void putValue(TableField<R, V> field, Field<V> value) {
        values.put(field, value);
    }

    @Override
    public String toString() {
        return "JooqUpdateBuilder{" +
                "idField=" + idField +
                ", values=" + values +
                ", changedIds=" + changedIds +
                '}';
    }

    /**
     * @param appliedChanges набор объектов с применёнными изменениями
     * @return {@link Multimap}, ставящий свойству набор объектов, у которых это свойство изменилось
     */
    @Nonnull
    private Multimap<ModelProperty, M> getPropertyToModelMultimap(
            Collection<? extends AppliedChanges<? extends M>> appliedChanges) {
        HashMultimap<ModelProperty, M> result = HashMultimap.create();
        for (var appliedChange : appliedChanges) {
            M modelObject = appliedChange.getModel();
            for (ModelProperty property : appliedChange.getPropertiesForUpdate()) {
                result.put(property, modelObject);
            }
        }
        return result;
    }
}
