package ru.yandex.direct.jooqmapperhelper;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.ToLongFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqModelToDbFieldValuesMapper;
import ru.yandex.direct.jooqmapper.kotlinwrite.KtModelToDbFieldValuesMapper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.KtModelChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;

/**
 * Хэлпер для удобства выполнения update'ов в базу с помощью {@link JooqModelToDbFieldValuesMapper}
 */
@ParametersAreNonnullByDefault
public class UpdateHelper<R extends Record> {
    private final DSLContext dslContext;
    private final TableField<R, Long> idField;
    private final Map<TableField<R, Object>, Map<Long, Object>> fieldToChanges;

    public UpdateHelper(DSLContext dslContext, TableField<R, Long> idField) {
        this.dslContext = dslContext;
        this.idField = idField;
        this.fieldToChanges = new HashMap<>();
    }

    public <M extends Model> UpdateHelper<R> processUpdateAll(JooqModelToDbFieldValuesMapper<M> jooqMapper,
                                                              Collection<AppliedChanges<M>> allChanges,
                                                              ToLongFunction<M> idGetter) {
        var updatedAtLeastInOneModelProperties = flatMapToSet(allChanges, AppliedChanges::getPropertiesForUpdate);
        allChanges.forEach(changes -> {
            Set<ModelProperty<? super M, ?>> propertiesForUpdate = changes.getPropertiesForUpdate();
            Map<TableField<R, ?>, ?> fieldValues = jooqMapper.getDbFieldValues(changes.getModel(), idField.getTable(),
                    propertiesForUpdate, updatedAtLeastInOneModelProperties);
            processUpdate(fieldValues, idGetter.applyAsLong(changes.getModel()));
        });
        return this;
    }

    public <M extends ModelWithId> UpdateHelper<R> processUpdateAll(JooqModelToDbFieldValuesMapper<M> jooqMapper,
                                                                    Collection<AppliedChanges<M>> allChanges) {
        return processUpdateAll(jooqMapper, allChanges, ModelWithId::getId);
    }

    /**
     * В момент вызова метода запрос в базу не выполняется.
     * Все запросы выполгняются в момент вызова метода {@link UpdateHelper#execute()}
     */
    public UpdateHelper<R> processUpdate(Map<TableField<R, ?>, ?> fieldValues,
                                         Long modelId) {
        EntryStream.of(fieldValues)
                .forKeyValue((tableField, value) -> {
                    // Добавление в fieldToChanges происходит в этом методе и мы гарантируем, что данный каст валиден
                    @SuppressWarnings("unchecked")
                    Map<Long, Object> idToValue = fieldToChanges.computeIfAbsent((TableField<R, Object>) tableField,
                            tf -> new HashMap<>());
                    checkState(!idToValue.containsKey(modelId),
                            "Field: " + tableField.getName() + " set more than one time for model with id: " + modelId);
                    idToValue.put(modelId, value);
                });
        return this;
    }

    public <M> UpdateHelper<R> processUpdateAllKt(
            KtModelToDbFieldValuesMapper<M> jooqMapper,
            Collection<KtModelChanges<Long, M>> changes
    ) {
        changes.forEach(
                singleChange -> {
                    var fieldValues = jooqMapper.getDbFieldValues(singleChange, idField.getTable());
                    fieldValues.forEach((tableField, value) -> {
                        // Добавление в fieldToChanges происходит в этом методе и мы гарантируем, что данный каст
                        // валиден
                        @SuppressWarnings("unchecked")
                        Map<Long, Object> idToValue = fieldToChanges.computeIfAbsent((TableField<R, Object>) tableField,
                                tf -> new HashMap<>());
                        Long modelId = singleChange.getId();
                        checkState(!idToValue.containsKey(modelId),
                                "Field: " + tableField.getName() + " set more than one time for model with id: " + modelId);
                        idToValue.put(modelId, value);
                    });
                }
        );
        return this;
    }

    public <M extends ModelWithId> UpdateHelper<R> processUpdate(JooqModelToDbFieldValuesMapper<M> jooqMapper,
                                                                 AppliedChanges<M> changes) {
        Map<TableField<R, ?>, ?> fieldValues = jooqMapper.getDbFieldValues(changes.getModel(), idField.getTable(),
                changes.getPropertiesForUpdate());
        return processUpdate(fieldValues, changes.getModel().getId());
    }

    public int execute() {
        Map<TableField<R, Object>, Field<Object>> updateValues = EntryStream.of(fieldToChanges)
                .mapToValue((dataField, idToValueMap) -> JooqMapperUtils.makeCaseStatement(idField, dataField,
                        idToValueMap))
                .toMap();
        var changedIds = EntryStream.of(fieldToChanges)
                .values()
                .flatMap(m -> m.keySet().stream())
                .toSet();
        if (changedIds.isEmpty()) {
            return 0;
        }

        return dslContext.update(idField.getTable())
                .set(updateValues)
                .where(idField.in(changedIds))
                .execute();
    }
}
