package ru.yandex.direct.juggler;

import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutionException;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.netty.util.HashedWheelTimer;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.io.RuntimeIoException;

import static ru.yandex.direct.utils.JsonUtils.fromJson;

public class AsyncHttpJugglerClient implements JugglerClient {

    public static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(10);
    public static final Duration DEFAULT_READ_TIMEOUT = Duration.ofSeconds(20);
    public static final Duration DEFAULT_REQUEST_TIMEOUT = DEFAULT_CONNECT_TIMEOUT.plus(DEFAULT_READ_TIMEOUT);
    public static final String CLIENT_NAME = AsyncHttpJugglerClient.class.getCanonicalName();
    private static final Logger logger = LoggerFactory.getLogger(AsyncHttpJugglerClient.class);
    private static final int HTTP_STATUS_OK = 200;

    private final String eventsUrl;
    private String apiUrl;
    private final boolean isGateway;
    private boolean isNeedClose = false;
    private final int readTimeoutMillis;
    private final int requestTimeoutMillis;
    private final AsyncHttpClient asyncHttpClient;
    private final HashedWheelTimer asyncHttpClientTimer;

    public AsyncHttpJugglerClient(String eventsUrl, boolean isGateway, AsyncHttpClient asyncHttpClient,
                                  Duration readTimeout, Duration requestTimeout) {
        this.asyncHttpClient = asyncHttpClient;
        this.asyncHttpClientTimer = null;
        this.readTimeoutMillis = Math.toIntExact(readTimeout.toMillis());
        this.requestTimeoutMillis = Math.toIntExact(requestTimeout.toMillis());
        this.isGateway = isGateway;
        this.eventsUrl = eventsUrl;
    }

    AsyncHttpJugglerClient(String eventsUrl, boolean isGateway, AsyncHttpClient asyncHttpClient,
                                  Duration readTimeout) {
        this(eventsUrl, isGateway, asyncHttpClient, readTimeout, DEFAULT_REQUEST_TIMEOUT);
    }

    public AsyncHttpJugglerClient(String eventsUrl, boolean isGateway, AsyncHttpClient asyncHttpClient) {
        this(eventsUrl, isGateway, asyncHttpClient, DEFAULT_READ_TIMEOUT);
    }

    public AsyncHttpJugglerClient(String eventsUrl, AsyncHttpClient asyncHttpClient) {
        this(eventsUrl, false, asyncHttpClient, DEFAULT_READ_TIMEOUT);
    }

    public AsyncHttpJugglerClient(String eventsUrl, boolean isGateway) {
        // Таймер тоже нужно закрывать, если мы его сами создаём (он работает с тредами)
        this.asyncHttpClientTimer = new HashedWheelTimer(
                new ThreadFactoryBuilder().setNameFormat("ahc-juggler-client-timer-%02d").setDaemon(true).build());
        DefaultAsyncHttpClientConfig.Builder builder = new DefaultAsyncHttpClientConfig.Builder();
        builder.setConnectTimeout(Math.toIntExact(DEFAULT_CONNECT_TIMEOUT.toMillis()));
        builder.setNettyTimer(this.asyncHttpClientTimer);
        this.asyncHttpClient = new DefaultAsyncHttpClient(builder.build());
        this.eventsUrl = eventsUrl;
        this.isGateway = isGateway;
        this.isNeedClose = true;
        this.readTimeoutMillis = Math.toIntExact(DEFAULT_READ_TIMEOUT.toMillis());
        this.requestTimeoutMillis = Math.toIntExact(DEFAULT_REQUEST_TIMEOUT.toMillis());
    }

    public AsyncHttpJugglerClient(String eventsUrl) {
        this(eventsUrl, false);
    }

    public void setApiUrl(String apiUrl) {
        this.apiUrl = apiUrl;
    }

    @Override
    public void sendEvents(List<JugglerEvent> events) {
        if (events.size() == 0) {
            return;
        }

        String path = isGateway ? "/events" : "/api/1/batch";
        String body = JsonUtils.toJson(isGateway ? new JugglerEventsBatch(CLIENT_NAME, events) : events);

        RequestBuilder builder = new RequestBuilder("POST")
                .setUrl(eventsUrl + path)
                .addHeader("Content-Type", "application/json")
                .setBody(body)
                .setRequestTimeout(requestTimeoutMillis)
                .setReadTimeout(readTimeoutMillis);
        Request request = builder.build();
        logger.debug("Request: {}\t{}", request, body);

        Response result;
        try {
            result = asyncHttpClient.executeRequest(request).get();
        } catch (ExecutionException e) {
            throw new JugglerClientException("Error when Juggler client send events", e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }

        if (result.getStatusCode() != HTTP_STATUS_OK) {
            throw new JugglerClientException(
                    "Error when Juggler client send events: Juggler returned " + result.getStatusCode() + " " + result
                            .getStatusText());
        }

        JugglerResponse response;
        try {
            response = fromJson(result.getResponseBody(), JugglerResponse.class);
        } catch (IllegalArgumentException e) {
            throw new JugglerClientException("Error when Juggler client send events:", e);
        }

        logger.debug("Response: {}", result);

        List<JugglerResponse.FailedEvent> failedEvents = response.getFailedEvents(events);
        if (!failedEvents.isEmpty()) {
            throw new FailedEventsException(failedEvents);
        }
    }

    /**
     * Возвращает список {@link JugglerChecksStateItem} по заданному фильтру {@param filter}
     * Для вызова метода должен быть отпределен apiUrl через метод {@link #setApiUrl(String)}
     *
     * @throws IllegalStateException если apiUrl не определен
     */
    @Override
    public List<JugglerChecksStateItem> getChecksStateItems(JugglerChecksStateFilter filter) {
        if (apiUrl == null) {
            throw new IllegalStateException("API url not defined {juggler.api}");
        }
        String body = JsonUtils.toJson(new JugglerChecksStateWrapper(filter));

        RequestBuilder builder = new RequestBuilder("POST")
                .setUrl(apiUrl + "/v2/checks/get_checks_state")
                .addHeader("Content-Type", "application/json")
                .setBody(body)
                .setReadTimeout(readTimeoutMillis);
        Request request = builder.build();
        logger.debug("Request: {}\t{}", request, body);

        Response result;
        try {
            result = asyncHttpClient.executeRequest(request).get();
        } catch (ExecutionException e) {
            throw new JugglerClientException("Error when Juggler client gets checks state", e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }

        if (result.getStatusCode() != HTTP_STATUS_OK) {
            throw new JugglerClientException(
                    "Error when Juggler client gets checks state: Juggler returned " + result.getStatusCode() + " " + result
                            .getStatusText() + ", response body: " + result.getResponseBody());
        }

        JugglerChecksStateResponse response;
        try {
            response = fromJson(result.getResponseBody(), JugglerChecksStateResponse.class);
        } catch (IllegalArgumentException e) {
            throw new JugglerClientException("Error when Juggler client gets checks state:", e);
        }
        logger.debug("Response: {}", response);
        return response.getItems();
    }

    @Override
    public void close() {
        if (isNeedClose) {
            try {
                asyncHttpClient.close();
                try {
                    asyncHttpClientTimer.stop();
                } catch (RuntimeException e) {
                    logger.warn("Unexpected error on HashedWheelTimer close", e);
                }
            } catch (IOException e) {
                throw new RuntimeIoException(e);
            }
        }
    }
}
