package ru.yandex.direct.juggler;

import java.time.Duration;
import java.util.List;

import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.MonotonicClock;
import ru.yandex.direct.utils.Transient;

import static ru.yandex.direct.juggler.JugglerAsyncSender.DEFAULT_CLOCK;
import static ru.yandex.direct.juggler.JugglerAsyncSender.DEFAULT_RETRY_DELAY;
import static ru.yandex.direct.juggler.JugglerAsyncSender.DEFAULT_SEND_DELAY;

/**
 * Асинхронный отправлятель событий в juggler.
 * <p>
 * Это то же, что и JugglerAsyncSender, но отправка событий происходит не на один,
 * а сразу на несколько указанных juggler-серверов.
 * <p>
 * Чтобы объяснить, зачем это нужно, процитирую ppalex@:
 * > с точки зрения кластера juggler-серверов - каждый обрабатывает сырые события
 * > независимо. Ни сами сырые событие, ни результат аггрегации не шарятся между
 * > серверами. Отправку уведомлений о событиях (golem, powny, прочее) - делает только
 * > один сервер, тот что является лидером.
 * >
 * > Я поставил эксперимент - завел проверку с небольшим ttl, отправил сырое событие на
 * > все три сервера, а затем стал отправлять его только на два (всех кроме мастера).
 * > В результате спустя ttl проверка "загорелась", при том что два других сервера
 * > считали ее "зеленой". отправка сырого события на мастер - вывела проверку из CRIT
 * > в OK.
 * >
 * > О том, что отказоустойчивость устроена именно так (посылайте одинаковое во все
 * > сервера) говорится на https://wiki.yandex-team.ru/sm/juggler/fault-tolerance/
 * > (во втором абзаце).
 */
public class JugglerAsyncMultiSender implements AutoCloseable, JugglerSender {
    private AutoCloseableList<JugglerAsyncSender> senders;

    public JugglerAsyncMultiSender(List<JugglerClient> clients) {
        this(clients, DEFAULT_SEND_DELAY, DEFAULT_RETRY_DELAY, 1000, DEFAULT_CLOCK);
    }

    public JugglerAsyncMultiSender(List<JugglerClient> clients, Duration minSendDelay, Duration retryDelay,
                                   int batchSize, MonotonicClock clock) {
        try (Transient<AutoCloseableList<JugglerAsyncSender>> transientSenders = new Transient<>()) {
            senders = transientSenders.item = new AutoCloseableList<>();
            for (JugglerClient client : clients) {
                try (Transient<JugglerAsyncSender> sender = new Transient<>()) {
                    sender.item = new JugglerAsyncSender(client, minSendDelay, retryDelay, batchSize, clock);
                    senders.add(sender);
                }
            }
            transientSenders.success();
        }
    }

    @Override
    public void sendEvent(JugglerEvent event, Duration timeout) {
        for (JugglerAsyncSender sender : senders) {
            sender.sendEvent(event, timeout);
        }
    }

    @Override
    public void close() {
        senders.close();
    }
}
