package ru.yandex.direct.juggler;

import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.AsyncConsumer;
import ru.yandex.direct.utils.BlockingChannelReader;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.MonotonicClock;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

/**
 * Асинхронный отправитель событий в juggler.
 * JugglerAsyncSender.sendEvent ставит событие в очередь на отправку не блокируя вызывающий тред.
 * Фактическая отправка происходит в отдельном треде.
 * <p>
 * В случае juggler-кластера, следует использовать JugglerAsyncMultiSender.
 */
public class JugglerAsyncSender implements AutoCloseable, JugglerSender {
    private static final Logger logger = LoggerFactory.getLogger(JugglerAsyncSender.class);

    public static final Duration DEFAULT_SEND_DELAY = Duration.ofMillis(200);
    public static final Duration DEFAULT_RETRY_DELAY = Duration.ofSeconds(10);
    public static final int DEFAULT_BATCH_SIZE = 100;
    public static final MonotonicClock DEFAULT_CLOCK = NanoTimeClock.CLOCK;

    private AsyncConsumer<JugglerEventWithDeadline> eventsConsumer;
    private MonotonicClock clock;

    public JugglerAsyncSender(JugglerClient client) {
        this(client, DEFAULT_SEND_DELAY, DEFAULT_RETRY_DELAY, DEFAULT_BATCH_SIZE, DEFAULT_CLOCK);
    }

    public JugglerAsyncSender(JugglerClient client, Duration minSendDelay, Duration retryDelay, int batchSize,
                              MonotonicClock clock) {
        this.clock = clock;
        this.eventsConsumer = new AsyncConsumer<>(
                (channelReader, ignored) -> run(channelReader, client, minSendDelay, retryDelay, batchSize, clock),
                Integer.MAX_VALUE,
                "Juggler-Sender",
                new JugglerEventQueue(clock)
        );
    }

    private static void run(BlockingChannelReader<JugglerEventWithDeadline> channelReader, JugglerClient client,
                            Duration minSendDelay, Duration retryDelay, int batchSize, MonotonicClock clock)
            throws InterruptedException {

        JugglerEventQueue allEvents = new JugglerEventQueue(clock);
        MonotonicTime lastSendStart = clock.getTime().minus(minSendDelay);

        while (!channelReader.takeAllInto(allEvents).isEmpty() || !channelReader.isClosed()) {

            Duration sendDelay = minSendDelay.minus(clock.getTime().minus(lastSendStart));
            clock.sleep(sendDelay);
            lastSendStart = clock.getTime();
            int timedOutEventsNum = allEvents.popTimedOutEvents().size();
            if (timedOutEventsNum > 0) {
                logger.warn("Timed out " + timedOutEventsNum + " events for " + client);
            }
            List<JugglerEvent> events = allEvents.stream()
                    .limit(batchSize)
                    .map(JugglerEventWithDeadline::getEvent)
                    .collect(Collectors.toList());

            try {
                client.sendEvents(events);
                events.forEach(event -> allEvents.dropKey(JugglerSubject.fromEvent(event)));
                logger.trace("Sent " + events.size() + " events to " + client);
            } catch (JugglerClient.FailedEventsException exc) {
                events.forEach(event -> allEvents.dropKey(JugglerSubject.fromEvent(event)));
                logger.error("Discarded " + exc.getFailedEvents().size() + " events out of " + events.size() + " by "
                        + client, exc);
            } catch (InterruptedRuntimeException exc) {
                logger.error("Juggler sending event process was interrupted", exc);
                break;
            } catch (Exception exc) {
                logger.warn("Failed to send " + events.size() + " events to " + client +
                        ", will retry in " + retryDelay, exc);
                channelReader.awaitClose(retryDelay);

                if (channelReader.isClosed()) {
                    // сервер не отвечает, приложение завершается, ждать смысла нет, сдаемся
                    channelReader.takeAllInto(allEvents);
                    logger.warn("AsyncSender is closed, " +
                            "giving up trying to send " + allEvents.size() + " events to " + client, exc);
                    break;
                }
            }
        }
        logger.info("No more events from channelReader");
    }

    /**
     * @param event   отправляемое событие
     * @param timeout если событие не было успешно отправлено в juggler за timeout,
     *                об этом появится запись в логе и событие будет выброшено из очереди.
     */
    @Override
    public void sendEvent(JugglerEvent event, Duration timeout) {
        eventsConsumer.acceptNonBlocking(new JugglerEventWithDeadline(event, clock.getTime().plus(timeout)));
    }

    @Override
    public void close() {
        logger.warn("JugglerAsyncSender was closed");
        eventsConsumer.close();
    }
}
