package ru.yandex.direct.juggler;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.direct.utils.LinkedHashMapQueue;
import ru.yandex.direct.utils.MonotonicClock;
import ru.yandex.direct.utils.MonotonicTime;

/**
 * Очередь событий для juggler. Используется в JugglerAsyncSender и JugglerAsyncMultiSender.
 * <p>
 * Особенности этой очереди:
 * <p>
 * - для конкретного хоста-сервиса хранится только последнее записанное событие. Это позволяет избежать
 * неограниченного роста потребляемой памяти и блокировки треда пользователя в ситуации, когда отправка
 * событий на juggler-сервер происходит медленнее, чем приходят новые события от сервисов.
 * <p>
 * - это FIFO, но, при записи нового события, если в очереди уже есть событие с такой же парой хост-сервис,
 * новое событие встанет не в конец очереди, а на место предыдущего.
 * <p>
 * - в очереди лежат не просто события, а события с дедлайном (JugglerEventWithDeadline),
 * просроченные события можно выбрасывать из очереди, вызывая popTimedOutEvents().
 */
public class JugglerEventQueue extends LinkedHashMapQueue<JugglerSubject, JugglerEventWithDeadline> {
    private MonotonicClock clock;

    public JugglerEventQueue(MonotonicClock clock) {
        super(eventWithDeadline -> JugglerSubject.fromEvent(eventWithDeadline.getEvent()));
        this.clock = clock;
    }

    public List<JugglerEventWithDeadline> popTimedOutEvents() {
        List<JugglerEventWithDeadline> timedOutEvents = new ArrayList<>();
        MonotonicTime now = clock.getTime();

        for (JugglerEventWithDeadline event : values()) {
            if (event.isTimedOutAt(now)) {
                timedOutEvents.add(event);
            }
        }
        timedOutEvents.forEach(event -> dropKey(JugglerSubject.fromEvent(event.getEvent())));

        return timedOutEvents;
    }

    @Override
    public boolean offer(JugglerEventWithDeadline t) {
        if (t.isTimedOutAt(clock.getTime())) {
            dropKey(JugglerSubject.fromEvent(t.getEvent()));
            return true;
        } else {
            return super.offer(t);
        }
    }
}
