package ru.yandex.direct.juggler;

import java.util.Objects;

import ru.yandex.direct.utils.SystemUtils;

public class JugglerSubject {
    private final String host;
    private final String instance;
    private final String service;

    public JugglerSubject(String host, String service, String instance) {
        this.host = host;
        this.instance = instance;
        this.service = service;
    }

    public JugglerSubject(String host, String service) {
        this(host, service, JugglerEvent.DEFAULT_INSTANCE);
    }

    public JugglerSubject(String service) {
        this(SystemUtils.strictHostname(), service);
    }

    public static JugglerSubject forService(String service) {
        return new JugglerSubject(SystemUtils.strictHostname(), service);
    }

    public static JugglerSubject fromEvent(JugglerEvent event) {
        return new JugglerSubject(event.getHost(), event.getService(), event.getInstance());
    }

    public JugglerEvent makeEvent(JugglerStatus status, String description) {
        return new JugglerEvent(host, service, instance, status, description);
    }

    public JugglerEvent makeOk() {
        return makeOk("OK");
    }

    public JugglerEvent makeOk(String description) {
        return makeEvent(JugglerStatus.OK, description);
    }

    public JugglerEvent makeCrit(String description) {
        return makeEvent(JugglerStatus.CRIT, description);
    }

    public JugglerEvent makeWarn(String description) {
        return makeEvent(JugglerStatus.WARN, description);
    }

    public JugglerEvent makeInfo(String description) {
        return makeEvent(JugglerStatus.INFO, description);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JugglerSubject that = (JugglerSubject) o;
        return Objects.equals(host, that.host) &&
                Objects.equals(instance, that.instance) &&
                Objects.equals(service, that.service);
    }

    @Override
    public int hashCode() {
        return Objects.hash(host, instance, service);
    }
}
