package ru.yandex.direct.libs.keywordutils.inclusion;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordForInclusion;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.inclusion.model.OrderedKeywordWithLemmas;
import ru.yandex.direct.libs.keywordutils.inclusion.model.SingleKeywordWithLemmas;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@SuppressWarnings("WeakerAccess")
@ParametersAreNonnullByDefault
public class KeywordInclusionUtils {

    /**
     * @return набор тех минус-фраз из {@code minusKeywords}, которые полностью вычитают
     * хотябы одну плюс-фразу из переданных plusKeywords {@code plusKeywords}
     * Пример:
     * <pre>
     *     {@code getIncludedMinusKeywords(asList("ёжик в тумане"), asList("ёжик", "туман", "серый ёжик"))}
     * </pre>
     * вернёт коллекцию {@code ("ёжик", "туман")}.
     */
    @Nonnull
    public static Set<String> getIncludedMinusKeywords(KeywordWithLemmasFactory keywordFactory,
                                                       StopWordMatcher stopWordMatcher,
                                                       Collection<String> plusKeywords, Collection<String> minusKeywords) {
        int objCount = plusKeywords.size() + minusKeywords.size();
        try (TraceProfile profile = Trace.current()
                .profile("keyword:inclusion", "minus_words", objCount)) {
            IdentityHashMap<KeywordForInclusion, String> minusKeywordForInclusions =
                    buildKeywordsByStrings(keywordFactory, minusKeywords);
            IdentityHashMap<KeywordForInclusion, String> plusKeywordsForInclusions =
                    buildKeywordsByStrings(keywordFactory, plusKeywords);

            Map<KeywordForInclusion, List<KeywordForInclusion>> plusKeywordsFullyIncluded =
                    getPlusKeywordsGivenEmptyResult(stopWordMatcher, plusKeywordsForInclusions.keySet(),
                            minusKeywordForInclusions.keySet());

            return StreamEx.ofValues(plusKeywordsFullyIncluded)
                    .flatMap(Collection::stream)
                    .map(minusKeywordForInclusions::get)
                    .toSet();
        }
    }

    /**
     * @return List<Keyword> набор минус фраз, которые пересекаются с плюс фразами слова {@code keyword}
     */
    public static List<Keyword> getIntersectedPlusWords(KeywordWithLemmasFactory keywordFactory,
                                                        StopWordMatcher stopWordMatcher,
                                                        KeywordWithMinuses keyword) {
        KeywordForInclusion plusKeyword = keywordFactory.keywordFrom(keyword.getKeyword());
        List<KeywordForInclusion> minusKeywords = mapList(keyword.getMinusKeywords(), keywordFactory::keywordFrom);
        Map<KeywordForInclusion, List<KeywordForInclusion>> intersected
                = getPlusKeywordsGivenEmptyResult(stopWordMatcher, singletonList(plusKeyword), minusKeywords);
        return intersected.values()
                .stream()
                .flatMap(k -> k.stream().map(KeywordForInclusion::getOrigin))
                .collect(Collectors.toList());
    }

    private static IdentityHashMap<KeywordForInclusion, String> buildKeywordsByStrings(
            KeywordWithLemmasFactory keywordWithLemmasFactory, Collection<String> minusKeywords) {
        return StreamEx.of(minusKeywords)
                .mapToEntry(identity())
                .mapKeys(keywordWithLemmasFactory::keywordFrom)
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * @return {@link Map}'у с теми плюс-фразами, которые полностью вычитаются той или иной минус-фразой.
     * Ключ &ndash; плюс-фраза, значение &ndash; минус-фразы, которые её перекрывают.
     */
    @Nonnull
    public static Map<KeywordForInclusion, List<KeywordForInclusion>> getPlusKeywordsGivenEmptyResult(
            StopWordMatcher stopWordMatcher,
            Collection<KeywordForInclusion> plusKeywords,
            Collection<KeywordForInclusion> minusKeywords) {
        int objCount = plusKeywords.size() + minusKeywords.size();
        try (TraceProfile profile = Trace.current()
                .profile("keyword:inclusion", "bulk_search", objCount)) {
            Map<String, List<KeywordForInclusion>> minusKeywordsByLemmas = StreamEx.of(minusKeywords)
                    .mapToEntry(KeywordForInclusion::allLemmas)
                    .flatMapValues(Collection::stream)
                    .invert()
                    .grouping();

            Map<KeywordForInclusion, List<KeywordForInclusion>> result = new HashMap<>();
            for (KeywordForInclusion plusKeyword : plusKeywords) {
                Set<String> lemmas = plusKeyword.allLemmas();
                Collection<KeywordForInclusion> matchingMinusKeywords =
                        StreamEx.of(lemmas).flatCollection(minusKeywordsByLemmas::get).distinct().toList();
                for (KeywordForInclusion minusKeyword : matchingMinusKeywords) {
                    if (isEmptyResultForKeywords(stopWordMatcher, plusKeyword, minusKeyword)) {
                        result.computeIfAbsent(plusKeyword, unused -> new ArrayList<>()).add(minusKeyword);
                    }
                }
            }

            return result;
        }
    }

    /**
     * Метод даёт ответ на вопрос "Даёт ли применение минус-фразы {@code minusKeyword} к фразе {@code plusKeyword} пустой результат?"
     *
     * @return {@code true}, если множество поисковых фраз, которые соответствуют {@code plusKeyword},
     * но не соответствуют {@code minusKeyword}, является пустым.
     */
    public static boolean isEmptyResultForKeywords(StopWordMatcher stopWordMatcher,
                                                   KeywordForInclusion plusKeyword, KeywordForInclusion minusKeyword) {

        if (minusKeyword.isExact() && !plusKeyword.isExact()) {
            // 'exact' не может включать 'not exact'
            return false;
        }

        if (minusKeyword.getOrderedKeywords().isEmpty() && minusKeyword.getSingleKeywords().isEmpty()) {
            // если в minusKeyword пусто, то он не может ничего включать
            return false;
        }

        // Если minusKeyword требуем, чтобы все леммы из plusKeyword были в minusKeyword
        boolean exactMatch = minusKeyword.isExact();
        Set<String> minusLemmas = minusKeyword.allLemmas();
        Set<String> plusLemmas = plusKeyword.allLemmas();
        if (exactMatch && !minusLemmas.containsAll(plusLemmas)) {
            // Для точного соответствия требуется, чтобы все plusLemmas включались в minusLemmas.
            // Если есть лишняя лемма, то возвращаем false
            return false;
        }

        // Quick win, если в minusKeyword находится singleKeyword, ни одна из лемм которого не совпадает с леммами plusKeyword
        outerLoop:
        for (SingleKeywordWithLemmas kw : minusKeyword.getAllSingleKeywords()) {
            for (String l : kw.getLemmas()) {
                if (plusLemmas.contains(l)) {
                    continue outerLoop;
                }
            }
            // нашли такое слово, все леммы которого не включаются в plusLemmas
            return false;
        }

        // Предобработка keyword'ов. Трактуем orderedKeyword с одним элементом как singleKeyword
        Collection<OrderedKeywordWithLemmas> minusOrderedKeywords = minusKeyword.getOrderedKeywords();
        Collection<SingleKeywordWithLemmas> minusSingleKeywords = minusKeyword.getSingleKeywords();
        Set<OrderedKeywordWithLemmas> singleElementOrderedKeywords =
                StreamEx.of(minusOrderedKeywords)
                        .filter(ok -> ok.getSingleKeywords().size() == 1)
                        .toSet();
        if (!singleElementOrderedKeywords.isEmpty()) {
            minusOrderedKeywords = StreamEx.of(minusOrderedKeywords)
                    .remove(singleElementOrderedKeywords::contains)
                    .toList();
            minusSingleKeywords = StreamEx.of(minusSingleKeywords)
                    .append(StreamEx.of(singleElementOrderedKeywords)
                            .flatCollection(OrderedKeywordWithLemmas::getSingleKeywords))
                    .toList();
        }

        // Проверяем, что все minus.orderedKeywords кого-то включают
        if (!areEmptyResultsForEveryOrderedKeywords(plusKeyword.getOrderedKeywords(), minusOrderedKeywords)) {
            return false;
        }
        // всем minus.orderedKeyword нашли соответствия в plus.orderedKeywords

        // Короткий путь для случая, когда нет minus.singleKeywords
        if (minusSingleKeywords.isEmpty()) {
            // minusOrderedKeywords не пуст, иначе бы мы уже вернули false в начале
            return true;
        }

        //все плюс-слова за исключением незафиксированных стоп-слов
        Set<SingleKeywordWithLemmas> plusSingleKeywords = StreamEx.of(plusKeyword.getOrderedKeywords())
                .flatCollection(OrderedKeywordWithLemmas::getSingleKeywords)
                .append(StreamEx.of(plusKeyword.getSingleKeywords())
                        .remove(singleKeyword -> !plusKeyword.isExact() && singleKeyword.isRaw() && stopWordMatcher
                                .isStopWord(singleKeyword.getNormalizedForm()))).toSet();
        // Проверяем, что все minus.singleKeywords кого-то включают
        if (!areEmptyResultsForEverySingleKeywords(plusSingleKeywords, minusSingleKeywords)) {
            return false;
        }
        // всем orderedKeywords и singleKeywords из minusKeyword нашлось соответствие
        return true;
    }

    /**
     * @return {@code true}, если для каждого из {@code minusOrderedKeywords} найдётся элемент из
     * {@code plusOrderedKeywords}, с которым получится пустой результат
     */
    private static boolean areEmptyResultsForEveryOrderedKeywords(
            Collection<OrderedKeywordWithLemmas> plusOrderedKeywords,
            Collection<OrderedKeywordWithLemmas> minusOrderedKeywords) {
        for (OrderedKeywordWithLemmas minusOrderedKeyword : minusOrderedKeywords) {
            if (!isAtLeastOneEmptyResultForOrderedKeyword(plusOrderedKeywords, minusOrderedKeyword)) {
                // есть minus.orderedKeyword, который никого не включает
                return false;
            }
        }
        return true;
    }

    /**
     * @param plusSingleKeywords  все плюс-слова за исключением незафиксированных стоп-слов
     * @param minusSingleKeywords список minusSingleKeywords
     * @return {@code true}, если для каждого из {@code minusSingleKeywords} найдётся элемент из
     * {@code plusSingleKeywords}, с которым получится пустой результат
     */
    private static boolean areEmptyResultsForEverySingleKeywords(
            Collection<SingleKeywordWithLemmas> plusSingleKeywords,
            Collection<SingleKeywordWithLemmas> minusSingleKeywords) {
        Map<String, List<SingleKeywordWithLemmas>> plusSingleKeywordByLemmas =
                StreamEx.of(plusSingleKeywords)
                        .distinct()
                        .mapToEntry(SingleKeywordWithLemmas::getLemmas)
                        .flatMapValues(Collection::stream)
                        .invert()
                        .grouping();
        for (SingleKeywordWithLemmas minusSingleKeyword : minusSingleKeywords) {
            if (!isAtLeastOneEmptyResultForSingleKeyword(plusSingleKeywordByLemmas,
                    minusSingleKeyword)) {
                // есть minus.singleKeyword, который ничего не включает
                return false;
            }
        }
        return true;
    }

    /**
     * @return {@code true}, если при применении {@code minusOrderedKeyword} хотя бы к одному из значений
     * {@code plusOrderedKeywords} получается пустой результат
     */
    private static boolean isAtLeastOneEmptyResultForOrderedKeyword(
            Collection<OrderedKeywordWithLemmas> plusOrderedKeywords,
            OrderedKeywordWithLemmas minusOrderedKeyword) {
        for (OrderedKeywordWithLemmas plusOrderedKeyword : plusOrderedKeywords) {
            if (isEmptyResultForOrderedKeyword(plusOrderedKeyword, minusOrderedKeyword)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return {@code true}, если при применении {@code minusSingleKeyword} хотя бы к одному из значений
     * {@code plusSingleKeywordByLemmas} получается пустой результат
     */
    private static boolean isAtLeastOneEmptyResultForSingleKeyword(
            Map<String, List<SingleKeywordWithLemmas>> plusSingleKeywordByLemmas,
            SingleKeywordWithLemmas minusSingleKeyword) {

        // Для каждой леммы minusSingleKeyword выберем подходящие singleKeywords из plusSingleKeywordByLemmas
        List<SingleKeywordWithLemmas> plusSingleKeywordsForLemma =
                StreamEx.of(minusSingleKeyword.getLemmas())
                        .map(plusSingleKeywordByLemmas::get)
                        .nonNull()
                        .flatMap(Collection::stream)
                        .toList();

        for (SingleKeywordWithLemmas plusSingleKeyword : plusSingleKeywordsForLemma) {
            if (isFirstIncludedInSecond(plusSingleKeyword, minusSingleKeyword)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return {@code true}, если при применении {@code minusOrderedKeyword} к {@code plusOrderedKeyword} получается
     * пустой результат, иначе - {@code false}
     */
    private static boolean isEmptyResultForOrderedKeyword(OrderedKeywordWithLemmas plusOrderedKeyword,
                                                          OrderedKeywordWithLemmas minusOrderedKeyword) {
        List<SingleKeywordWithLemmas> otherSingleKeywords = minusOrderedKeyword.getSingleKeywords();
        List<SingleKeywordWithLemmas> plusSingleKeywords = plusOrderedKeyword.getSingleKeywords();
        if (otherSingleKeywords.size() > plusSingleKeywords.size()) {
            // Если в minusOrderedKeyword слов больше, то minusOrderedKeyword не может включать в себя plusOrderedKeyword
            return false;
        }
        for (int i = 0; i <= plusSingleKeywords.size() - otherSingleKeywords.size(); i++) {
            boolean allCount = true;
            for (int j = 0; j < otherSingleKeywords.size(); j++) {
                if (!isFirstIncludedInSecond(plusSingleKeywords.get(i + j), otherSingleKeywords.get(j))) {
                    allCount = false;
                    break;
                }
            }
            if (allCount) {
                // нашли подходящий i
                return true;
            }
        }
        return false;
    }

    /**
     * @return {@code true}, если первое слово полностью включается во второе
     */
    public static boolean isFirstIncludedInSecond(SingleKeywordWithLemmas first,
                                                  SingleKeywordWithLemmas second) {
        // если оба слова зафиксированы, то сравниваем их как есть не глядя на леммы
        if (first.isFixed() && second.isFixed()) {
            return first.getNormalizedForm().equals(second.getNormalizedForm());
        }

        // если второе слово зафиксировано, то второе точно не включает первое
        if (second.isFixed()) {
            return false;
        }

        Collection<String> otherWordLemmas = second.getLemmas().size() <= 10
                ? second.getLemmas()
                : new HashSet<>(second.getLemmas());

        // если второе не зафиксировано, а первое - зафиксировано, то второе будет включать данное
        // в том случае, если у них есть хотя бы одна общая лемма
        if (first.isFixed()) {
            return first.getLemmas().stream().anyMatch(otherWordLemmas::contains);
        }

        // если оба слова не зафиксированы, то все леммы второго слова должны содержать все леммы первого
        return otherWordLemmas.containsAll(first.getLemmas());
    }

}
