package ru.yandex.direct.libs.keywordutils.inclusion.model;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.libs.keywordutils.model.Keyword;

public class KeywordForInclusion {
    private final Keyword origin;
    private final boolean exact;
    private final Collection<OrderedKeywordWithLemmas> orderedKeywords;
    private final Collection<SingleKeywordWithLemmas> singleKeywords;
    private Set<String> allLemmas;
    private Set<SingleKeywordWithLemmas> allSingleKeywords;

    KeywordForInclusion(Keyword origin, boolean exact,
                        List<SingleKeywordWithLemmas> singleKeywords,
                        Collection<OrderedKeywordWithLemmas> orderedKeywords) {
        this.origin = origin;
        this.exact = exact;
        this.orderedKeywords = Collections.unmodifiableCollection(orderedKeywords);
        this.singleKeywords = Collections.unmodifiableCollection(singleKeywords);
    }

    public Set<String> allLemmas() {
        if (allLemmas == null) {
            allLemmas = StreamEx.of(getAllSingleKeywords())
                    .flatCollection(SingleKeywordWithLemmas::getLemmas)
                    .toSet();
        }
        return allLemmas;
    }

    public boolean isExact() {
        return exact;
    }

    public Collection<OrderedKeywordWithLemmas> getOrderedKeywords() {
        return orderedKeywords;
    }

    public Set<SingleKeywordWithLemmas> getAllSingleKeywords() {
        if (allSingleKeywords == null) {
            allSingleKeywords = StreamEx.of(this.orderedKeywords)
                    .flatCollection(OrderedKeywordWithLemmas::getSingleKeywords)
                    .append(this.singleKeywords).toSet();
        }
        return allSingleKeywords;
    }

    public Keyword getOrigin() {
        return origin;
    }

    public Collection<SingleKeywordWithLemmas> getSingleKeywords() {
        return singleKeywords;
    }

    public String originString() {
        return origin.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        KeywordForInclusion that = (KeywordForInclusion) o;
        return exact == that.exact &&
                Objects.equals(origin, that.origin) &&
                Objects.equals(orderedKeywords, that.orderedKeywords) &&
                Objects.equals(singleKeywords, that.singleKeywords);
    }

    @Override
    public int hashCode() {
        return Objects.hash(origin, exact, orderedKeywords, singleKeywords);
    }

    @Override
    public String toString() {
        String format = isExact() ? "\"%s\"" : "%s";
        return String.format(format,
                StreamEx.of(orderedKeywords)
                        .map(OrderedKeywordWithLemmas::toString)
                        .append(StreamEx.of(singleKeywords).map(SingleKeywordWithLemmas::toString))
                        .sorted()
                        .joining(" "));
    }
}
