package ru.yandex.direct.libs.keywordutils.inclusion.model;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.UncheckedExecutionException;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Collections.emptyList;

/**
 * Создаёт из обычных {@link AnyKeyword} ключевые фразы с леммами.
 */
@ParametersAreNonnullByDefault
public class KeywordWithLemmasFactory {

    private static final Logger logger = LoggerFactory.getLogger(KeywordWithLemmasFactory.class);

    private final ParseKeywordCache keywordCache;
    private final SingleKeywordsCache singleKeywordsCache;

    public KeywordWithLemmasFactory(int clemmerCacheCapacity,
                                    int keywordsCacheCapacity,
                                    int singleKeywordsCacheCapacity) {
        this(new SingleKeywordsCache(clemmerCacheCapacity, singleKeywordsCacheCapacity),
                new ParseKeywordCache(keywordsCacheCapacity));
    }

    public KeywordWithLemmasFactory(SingleKeywordsCache singleKeywordsCache, ParseKeywordCache keywordCache) {
        this.singleKeywordsCache = singleKeywordsCache;
        this.keywordCache = keywordCache;
    }

    public KeywordWithLemmasFactory() {
        this(new SingleKeywordsCache(), new ParseKeywordCache());
    }

    /**
     * По заданной фразе производит её лемматизацию и выдаёт экземпляр {@link KeywordForInclusion}.
     * <p>
     * Стоп-слова при преобразованиях сохраняются.
     *
     * @param keyword ключевая фраза для разбора и лемматизации
     * @return {@link KeywordForInclusion} представление фразы для вычисления включений
     */
    public KeywordForInclusion keywordFrom(String keyword) {
        return keywordFrom(keywordCache.getCached(keyword));
    }

    /**
     * По заданной фразе производит её лемматизацию и выдаёт экземпляр {@link KeywordForInclusion}.
     * <p>
     * Стоп-слова при преобразованиях сохраняются.
     *
     * @param keyword ключевая фраза для разбора и лемматизации
     * @return {@link KeywordForInclusion} представление фразы для вычисления включений
     */
    public KeywordForInclusion keywordFrom(Keyword keyword) {
        Map<? extends Class<?>, List<AnyKeyword>> classListMap =
                StreamEx.of(keyword.getAllKeywords())
                        .groupingBy(Object::getClass);

        @SuppressWarnings("unchecked")
        List<SingleKeyword> singleKeywords =
                firstNonNull((List) classListMap.get(SingleKeyword.class), emptyList());
        List<SingleKeywordWithLemmas> singleKeywordWithLemmas = StreamEx.of(singleKeywords)
                .flatCollection(sk -> singleKeywordsCache.singleKeywordsFrom(sk.getWord()))
                .toList();
        @SuppressWarnings("unchecked")
        List<OrderedKeyword> orderedKeywords =
                firstNonNull((List) classListMap.get(OrderedKeyword.class), emptyList());
        List<OrderedKeywordWithLemmas> orderedKeywordWithLemmas = StreamEx.of(orderedKeywords)
                .map(this::orderedKeywordFrom)
                .toList();
        return new KeywordForInclusion(keyword, keyword.isQuoted(), singleKeywordWithLemmas, orderedKeywordWithLemmas);
    }

    /**
     * Преобразует упорядоченный набор слов, добавляя к ним леммы.
     *
     * @param orderedKeyword {@link OrderedKeyword} упорядоченный набор слов
     * @return {@link OrderedKeywordWithLemmas} упорядоченный набор слов из фразы с леммами
     * @see {@link SingleKeywordFactory}
     */
    private OrderedKeywordWithLemmas orderedKeywordFrom(OrderedKeyword orderedKeyword) {
        List<SingleKeywordWithLemmas> singleKeywordWithLemmas = StreamEx.of(orderedKeyword.getSingleKeywords())
                .map(SingleKeyword::getWord)
                .flatCollection(singleKeywordsCache::singleKeywordsFrom)
                .toList();

        return new OrderedKeywordWithLemmas(singleKeywordWithLemmas);
    }

    /**
     * Exception-safe обёртка для {@link #keywordFrom(String)}.
     *
     * @see #keywordFrom(String)
     */
    public Optional<KeywordForInclusion> safeKeywordFrom(String keyword) {
        try {
            return Optional.of(keywordFrom(keyword));
        } catch (UncheckedExecutionException e) {
            logger.info("Can't parse keyword {}: {}", keyword, e.getMessage());
            return Optional.empty();
        }
    }
}
