package ru.yandex.direct.libs.keywordutils.inclusion.model;

import java.util.Comparator;
import java.util.List;

import javax.annotation.Nonnull;

import com.google.common.base.CharMatcher;
import one.util.streamex.StreamEx;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.direct.clemmer.ClemmerWord;
import ru.yandex.direct.libs.keywordutils.helper.ClemmerCache;
import ru.yandex.direct.libs.keywordutils.parser.SplitCompoundWordExpressionTransform;
import ru.yandex.direct.utils.TextConstants;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;

/**
 * Создаёт из обычных {@link Word} ключевые фразы с леммами.
 */
public class SingleKeywordFactory {
    private static final Comparator<String> STRING_LEXICOGRAPHIC_COMPARE =
            Comparator.comparingInt(String::length).thenComparing(String::compareTo);
    private static final CharMatcher CONTAINS_ANY_RUS_LETTERS = CharMatcher.anyOf(TextConstants.RUS_LETTERS);
    private static final CharMatcher CONTAINS_ANY_LAT_LETTERS = CharMatcher.anyOf(TextConstants.LAT_LETTERS);

    private final ClemmerCache clemmerCache;

    public SingleKeywordFactory() {
        this(new ClemmerCache());
    }

    public SingleKeywordFactory(ClemmerCache clemmerCache) {
        this.clemmerCache = clemmerCache;
    }

    /**
     * Разбивает переданное слово на леммы и возвращает набор {@link SingleKeywordWithLemmas}.
     * Перед этим разделяет сложные слова (напр. санкт-петербург -> санкт петербург)
     * <p>
     * {@code Clemmer} в некоторых случаях для одного {@link Word} находит несколько {@link ClemmerWord}.
     * Например, "R17.5" -> "R" + "17.5". Чтобы правильно обрабатывать подобные случаи при поиске включений,
     * мы для каждого полученного от Clemmer'а слова создаём {@link SingleKeywordWithLemmas}.
     */
    public List<SingleKeywordWithLemmas> singleKeywordsFrom(Word word) {
        Word lowerCasedWord = wordToLowerCase(word);

        // Не ходим в леммер если слово состоит из символов разного алфавита
        if (CONTAINS_ANY_LAT_LETTERS.matchesAnyOf(lowerCasedWord.getText()) &&
                CONTAINS_ANY_RUS_LETTERS.matchesAnyOf(lowerCasedWord.getText())) {
            return List.of(new SingleKeywordWithLemmas(new Word(word.getKind(),
                    lowerCasedWord.getText()),
                    List.of(lowerCasedWord.getText())));
        }
        List<String> texts = SplitCompoundWordExpressionTransform.splitRawWordRegexp(lowerCasedWord.getText());
        List<ClemmerWord> clemmerWords = flatMap(texts, clemmerCache::getCached);
        return StreamEx.of(clemmerWords)
                .map(cw -> new SingleKeywordWithLemmas(new Word(word.getKind(), cw.getText()),
                        sortLemmas(cw.getLemmas())))
                .toList();
    }

    @Nonnull
    private static Word wordToLowerCase(Word word) {
        // приводим текст слова к нижнему регистру, если требуется
        Word lowerCasedWord;
        boolean wordHasUpperCase = CharMatcher.javaUpperCase().matchesAnyOf(word.getText());
        if (wordHasUpperCase) {
            lowerCasedWord = new Word(word.getKind(), word.getText().toLowerCase());
        } else {
            lowerCasedWord = word;
        }
        return lowerCasedWord;
    }

    //сортируем леммы
    private static List<String> sortLemmas(List<String> lemmas) {
        return lemmas
                .stream()
                .sorted(STRING_LEXICOGRAPHIC_COMPARE)
                .collect(toList());
    }
}
