package ru.yandex.direct.libs.keywordutils.parser;

import java.util.List;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.advq.query.ast.Difference;
import ru.yandex.advq.query.ast.Expression;
import ru.yandex.advq.query.ast.ExpressionVisitor;
import ru.yandex.advq.query.ast.Intersection;
import ru.yandex.advq.query.ast.QuotedWords;
import ru.yandex.advq.query.ast.SquareBrackets;
import ru.yandex.advq.query.ast.Union;
import ru.yandex.advq.query.ast.Word;
import ru.yandex.advq.query.ast.WordsNum;

import static java.util.Collections.singletonList;

public class KeywordBuilderVisitor implements ExpressionVisitor<KeywordBuilder> {
    private static final Logger logger = LoggerFactory.getLogger(KeywordBuilderVisitor.class);

    private KeywordBuilder visitAll(List<? extends Expression> expressions) {
        return StreamEx.of(expressions)
                .map(e -> e.accept(this))
                .reduce((keywordAccumulator, keywordAccumulator2) -> keywordAccumulator2.addTo(keywordAccumulator))
                .orElseGet(KeywordBuilder::new);
    }

    /**
     * {@link Union} По сути это объединение ключевых фраз по "ИЛИ". Такое объединение сейчас ожидаем только
     * в правой части {@link Difference}, который обрабатывается явно в {@link KeywordWithMinusesBuilder}.
     * А также в старых кампаниях, когда валидация еще допускала объединение без разбиения на несколько фраз.
     * Для того, чтобы интерфейс старых кампаний не падал (см. DIRECT-145444), игнорируем оператор объединения.
     */
    @Override
    public KeywordBuilder visitUnion(Union expr) {
        logger.trace("visitUnion()");
        return visitAll(expr.getChildren());
    }

    /**
     * {@link Difference} явно обрабатывается в {@link KeywordWithMinusesBuilder}.
     * Если встречаем где-то ещё &ndash; падаем.
     */
    @Override
    public KeywordBuilder visitDifference(Difference expr) {
        logger.trace("visitDifference()");
        throw new UnsupportedOperationException();
    }

    /**
     * {@link WordsNum} в Директе не поддерживается
     */
    @Override
    public KeywordBuilder visitWordsNum(WordsNum expr) {
        logger.trace("visitWordsNum()");
        throw new UnsupportedOperationException();
    }

    /**
     * {@link QuotedWords} определяет кавычки вокруг фразы
     */
    @Override
    public KeywordBuilder visitQuotedWords(QuotedWords expr) {
        logger.trace("visitQuotedWords()");
        KeywordBuilder res = new KeywordBuilder().exact();
        visitAll(singletonList(expr.getChild())).addTo(res);
        return res;
    }

    /**
     * {@link Intersection} &ndash; родительский элемент для директовской фразы
     */
    @Override
    public KeywordBuilder visitIntersection(Intersection expr) {
        logger.trace("visitIntersection()");
        return visitAll(expr.getChildren());
    }

    /**
     * {@link SquareBrackets} квадратные скобочки
     */
    @Override
    public KeywordBuilder visitSquareBrackets(SquareBrackets expr) {
        logger.trace("visitSquareBrackets()");
        KeywordBuilder kb = new KeywordBuilder();
        kb.startBrackets();
        visitAll(expr.getWords()).addTo(kb);
        kb.endBrackets();
        return kb;
    }

    /**
     * {@link Word} отдельное слово
     */
    @Override
    public KeywordBuilder visitWord(Word expr) {
        logger.trace("visitWord()");
        return new KeywordBuilder().pushWord(expr);
    }
}
