package ru.yandex.direct.libs.keywordutils.parser;

import java.util.regex.Pattern;

import ru.yandex.advq.query.QueryParser;
import ru.yandex.advq.query.QuerySyntax;
import ru.yandex.advq.query.ast.Expression;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class KeywordParser {

    private static final Pattern REPLACE_QUOTE_REGEX = Pattern.compile("^'+");
    private static final Pattern SEARCH_QUOTE_REGEX = Pattern.compile("(\\s'+)");

    public static Keyword parse(String query) {
        try (TraceProfile profile = Trace.current().profile("keyword:parser", "parse")) {
            Expression expression = buildExpression(query);
            return parseExpression(expression);
        }
    }

    public static KeywordWithMinuses parseWithMinuses(String query) {
        try (TraceProfile profile = Trace.current().profile("keyword:parser", "parseWithMinuses")) {
            Expression expression = buildExpression(query);
            return KeywordWithMinusesBuilder.buildKeywordWithMinusesFrom(expression);
        }
    }

    public static Keyword parseExpression(Expression expression) {
        KeywordBuilder keywordBuilder = expression.accept(new KeywordBuilderVisitor());
        return keywordBuilder.build();
    }

    public static Expression buildExpression(String query) {
        query = cleanQuery(query);
        QueryParser parser =
                QueryParser.newBuilder(QuerySyntax.DIRECT)
                        .setSplitCompoundWords(false)
                        .setOptimize(false)
                        .setPrepareQueries(false)
                        .build();
        return replaceChars(parser.parseQuery(query));
    }

    private static String cleanQuery(String query) {
        query = REPLACE_QUOTE_REGEX.matcher(query).replaceFirst("");

        var matcher = SEARCH_QUOTE_REGEX.matcher(query);
        var queryBuilder = new StringBuilder(query);
        var deleted = 0;

        while (matcher.find()) {
            for (int i = 0; i < matcher.groupCount(); i++) {
                queryBuilder.delete(matcher.start(i) + 1 - deleted, matcher.end(i) - deleted);
                deleted += matcher.end(i) - matcher.start(i) - 1;
            }
        }

        return queryBuilder.toString().trim();
    }

    /**
     * Заменяет символы. Например, "ё" -> "е".
     */
    private static Expression replaceChars(Expression expression) {
        return expression.accept(ReplaceCharsExpressionTransform.INSTANCE);
    }
}
