package ru.yandex.direct.libs.keywordutils.parser;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;
import com.google.common.base.Splitter;

import ru.yandex.advq.query.ast.Expression;
import ru.yandex.advq.query.ast.ExpressionTransform;
import ru.yandex.advq.query.ast.Intersection;
import ru.yandex.advq.query.ast.Word;

/**
 * Разделяет составные слова (например санкт-петербург) на отдельные части
 * Отличия от advq:
 * Не считает разделителем "+"
 * Разделяет 123.124.123.125 -> 123 124 123 125 (advq делает 123.124 123.125)
 */
public class SplitCompoundWordExpressionTransform extends ExpressionTransform {
    public static final SplitCompoundWordExpressionTransform
            INSTANCE = new SplitCompoundWordExpressionTransform();

    private static final CharMatcher DELIMITER_CHARACTERS = CharMatcher.anyOf("-,#$@*%'");
    private static final CharMatcher DELIMITER_DOT = CharMatcher.is('.');

    private static final Splitter DELIMITER_CHARACTERS_SPLITTER = Splitter.on(DELIMITER_CHARACTERS).omitEmptyStrings();
    private static final Splitter DELIMITER_DOT_SPLITTER = Splitter.on(DELIMITER_DOT).omitEmptyStrings();
    private static final Pattern NUMBER_PATTERN = Pattern.compile("^!?\\d+\\.\\d+$");

    @Override
    public Expression visitWord(Word expr) {
        // Разбираем составные слова на отдельные части
        Intersection.Builder builder = new Intersection.Builder();
        List<String> textParts = splitRawWordRegexp(expr.getText());
        for (String textPart : textParts) {
            builder.add(new Word(expr.getKind(), textPart));
        }
        return builder.build();
    }

    /**
     * Этот метод используется в валидации, чтобы понять,
     * разделяется одно слово на несколько или нет.
     * Поэтому, разделение на несколько слов при парсинге должно работать
     * строго через этот метод.
     */
    public static List<String> splitRawWordRegexp(String text) {
        List<String> words = new ArrayList<>();
        //разбиваем слово по стандартным разделителям
        Iterable<String> parts = DELIMITER_CHARACTERS_SPLITTER.split(text);
        for (String part : parts) {
            if (part.indexOf('.') == -1 || NUMBER_PATTERN.matcher(part).matches()) {
                //если нет "." или это число, то добавляем как есть
                words.add(part);
            } else {
                // считаем "." разделителем и разбиваем на слова
                DELIMITER_DOT_SPLITTER.split(part).forEach(words::add);
            }
        }
        return words;
    }
}
