package ru.yandex.direct.libs.keywordutils.parser;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;


@RunWith(Parameterized.class)
public class KeywordParserTest {

    @Parameterized.Parameter
    public String input;

    @Parameterized.Parameter(1)
    public String expected;

    @Parameterized.Parameters(name = "parse({0}) == {1}")
    public static Collection<Object[]> params() {
        return Arrays.asList(new Object[][]{
                // простые кейсы
                {"конь", "конь"},
                {"коня", "коня"},
                {"!коня", "!коня"},
                {"!Коня", "!Коня"},
                {"+коня", "+коня"},
                {"+Коня", "+Коня"},
                {"коня1", "коня1"},
                {"Коня1", "Коня1"},
                {"коняrw", "коняrw"},
                {"коняRW", "коняRW"},
                {"конь вакуум", "конь вакуум"},
                {"кони +в вакууме", "кони +в вакууме"},
                {"кони +в !вакууме", "кони +в !вакууме"},
                {"!кони +в вакууме", "!кони +в вакууме"},
                {"123 коня", "123 коня"},
                {"коня 123", "коня 123"},
                {"коня 123.456", "коня 123.456"},
                {"кони +в !вакууме куплю", "кони +в !вакууме куплю"},

                //замена символов
                {"ёлка", "елка"},
                {"Ёлка", "Елка"},
                {"[ёлка]", "[елка]"},
                {"[Ёлка]", "[Елка]"},
                {"\"ёлка\"", "\"елка\""},
                {"\"Ёлка\"", "\"Елка\""},
                {"да ёшкин ты конь", "да ешкин ты конь"},
                {"да ЁШКИН ты конь", "да ЕШКИН ты конь"},


                //не разделяем составные слова
                {"Санкт-Петербург", "Санкт-Петербург"},
                {"Санкт-Петербург-Москва", "Санкт-Петербург-Москва"},
                {"!Санкт-Петербург-Москва", "!Санкт-Петербург-Москва"},
                {"красно-черный", "красно-черный"},
                {"красно-черного", "красно-черного"},
                {"!красно-черного", "!красно-черного"},
                {"продать +красно-черного коня", "продать +красно-черного коня"},
                {"123.456", "123.456"},
                {"123.456.789", "123.456.789"},
                {"123.456.789.124", "123.456.789.124"},
                {"купить +123.123", "купить +123.123"},
                {"купить +123.123.345", "купить +123.123.345"},
                {"123.f", "123.f"},
                {"123.f.34", "123.f.34"},
                {"а.б", "а.б"},
                {"Доставка.Быстро", "Доставка.Быстро"},

                //не разделяем составные слова по точке (в конце)
                {"а.", "а."},
                {"а. ", "а."},
                {"а. б", "а. б"},
                {"1.", "1."},
                {"1. ", "1."},
                {"1. а", "1. а"},
                {"Цена.", "Цена."},
                {"Билеты в Грецию. Цена", "Билеты в Грецию. Цена"},
                {"Playstation 3. Цена", "Playstation 3. Цена"},

                //не разделяем составные слова по точке (в середине и в конце)
                {"а.б.", "а.б."},
                {"а.б. ", "а.б."},
                {"а.б. в", "а.б. в"},
                {"1.2.", "1.2."},
                {"1.2. ", "1.2."},
                {"1.2. а", "1.2. а"},
                {"123.456.789.124.", "123.456.789.124."},
                {"+123.456.789.124.", "+123.456.789.124."},
                {"Яндекс.Директ.", "Яндекс.Директ."},
                {"Яндекс.Директ. Настройка", "Яндекс.Директ. Настройка"},

                //не разделяем составные слова по другим разделителям
                {"Hemingway's Hotel", "Hemingway's Hotel"},
                {"123f123", "123f123"},
                {"123+214", "123+214"}, // не разбиваем слово с плюсом
                {"c++", "c++"}, // два плюса допустимо. оставляем как есть

                //не разделяем составные слова по нескольким разделителям
                {"WAAS-VB-NAM-4.2", "WAAS-VB-NAM-4.2"},
                {"Санкт-Петербург. Цена", "Санкт-Петербург. Цена"},
                {"!Санкт-Петербург. Цена", "!Санкт-Петербург. Цена"},
                {"+Санкт-Петербург. Цена", "+Санкт-Петербург. Цена"},

                {"красно,черный", "красно,черный"},
                {"красно#черный", "красно#черный"},
                {"красно$черный", "красно$черный"},
                {"красно@черный", "красно@черный"},
                {"красно*черный", "красно*черный"},
                {"красно%черный", "красно%черный"},

                // кейсы с квадратными скобками
                {"[конь]", "[конь]"},
                {"[коня]", "[коня]"},
                {"[!коня]", "[!коня]"},
                {"[!Коня]", "[!Коня]"},
                {"[+коня]", "[+коня]"},
                {"[+Коня]", "[+Коня]"},
                {"[коня1]", "[коня1]"},
                {"[Коня1]", "[Коня1]"},
                {"[коняrw]", "[коняrw]"},
                {"[коняRW]", "[коняRW]"},

                {"[конь вакуум]", "[конь вакуум]"},
                {"[кони +в вакууме]", "[кони +в вакууме]"},
                {"[кони +в !вакууме]", "[кони +в !вакууме]"},
                {"[!кони +в вакууме]", "[!кони +в вакууме]"},
                {"[кони +в !вакууме куплю]", "[кони +в !вакууме куплю]"},

                //не разделяем составные слова по дефису
                {"[Санкт-Петербург]", "[Санкт-Петербург]"},
                {"[!Санкт-Петербург-Москва]", "[!Санкт-Петербург-Москва]"},
                {"[продать +красно-черного коня]", "[продать +красно-черного коня]"},

                //не разделяем составные слова по точке (в середине)
                {"[123.456]", "[123.456]"},

                //не разделяем составные слова по точке (в конце)
                {"[а.]", "[а.]"},
                {"[а. ]", "[а.]"},
                {"[а. б]", "[а. б]"},
                {"[Билеты в Грецию. Цена]", "[Билеты в Грецию. Цена]"},
                {"[Билеты в Грецию.] Цена", "[Билеты в Грецию.] Цена"},
                {"[Playstation 3.] Цена", "[Playstation 3.] Цена"},

                // комбинированные кейсы с квадратными скобками
                {"кони [в вакууме]", "кони [в вакууме]"},
                {"[в вакууме] кони", "[в вакууме] кони"},
                {"кони [сферические в вакууме]", "кони [сферические в вакууме]"},
                {"[сферические в вакууме] кони", "[сферические в вакууме] кони"},
                {"!кони [!сферические в !вакууме]", "!кони [!сферические в !вакууме]"},
                {"[!сферические в !вакууме] !кони", "[!сферические в !вакууме] !кони"},
                {"+кони [+сферические в !вакууме]", "+кони [+сферические в !вакууме]"},
                {"[+сферические в !вакууме] +кони", "[+сферические в !вакууме] +кони"},

                {"[сферические в !вакууме] кони [с яблоками]", "[сферические в !вакууме] кони [с яблоками]"},

                // кейсы с кавычками
                {"\"конь\"", "\"конь\""},
                {"\"коня\"", "\"коня\""},
                {"\"!коня\"", "\"!коня\""},
                {"\"!Коня\"", "\"!Коня\""},
                {"\"+коня\"", "\"+коня\""},
                {"\"+Коня\"", "\"+Коня\""},
                {"\"коня1\"", "\"коня1\""},
                {"\"Коня1\"", "\"Коня1\""},
                {"\"коняrw\"", "\"коняrw\""},
                {"\"коняRW\"", "\"коняRW\""},
                {"\"конь вакуум\"", "\"конь вакуум\""},
                {"\"кони +в вакууме\"", "\"кони +в вакууме\""},
                {"\"кони +в !вакууме\"", "\"кони +в !вакууме\""},
                {"\"!кони +в вакууме\"", "\"!кони +в вакууме\""},
                {"\"кони +в !вакууме куплю\"", "\"кони +в !вакууме куплю\""},


                //не разделяем составные слова по дефису
                {"\"Санкт-Петербург\"", "\"Санкт-Петербург\""},
                {"\"!Санкт-Петербург-Москва\"", "\"!Санкт-Петербург-Москва\""},
                {"\"продать +красно-черного коня\"", "\"продать +красно-черного коня\""},

                //не разделяем составные слова по точке (в середине)
                {"\"123.456\"", "\"123.456\""},

                //не разделяем составные словав по точке (в конце)
                {"\"а.\"", "\"а.\""},
                {"\"а. \"", "\"а.\""},
                {"\"а. б\"", "\"а. б\""},
                {"\"Билеты в Грецию. Цена\"", "\"Билеты в Грецию. Цена\""},

                // кавычки + квадратные скобки
                {"\"кони [в вакууме]\"", "\"кони [в вакууме]\""},
                {"\"[в вакууме] кони\"", "\"[в вакууме] кони\""},
                {"\"кони [сферические в вакууме]\"", "\"кони [сферические в вакууме]\""},
                {"\"[сферические в вакууме] кони\"", "\"[сферические в вакууме] кони\""},
                {"\"!кони [!сферические в !вакууме]\"", "\"!кони [!сферические в !вакууме]\""},
                {"\"[!сферические в !вакууме] !кони\"", "\"[!сферические в !вакууме] !кони\""},
                {"\"+кони [+сферические в !вакууме]\"", "\"+кони [+сферические в !вакууме]\""},
                {"\"[+сферические в !вакууме] +кони\"", "\"[+сферические в !вакууме] +кони\""},

                {"\"[сферические в !вакууме] кони [с яблоками]\"", "\"[сферические в !вакууме] кони [с яблоками]\""},
        });
    }

    @Test
    public void test() {
        String actual = String.valueOf(KeywordParser.parse(input));
        assertThat(actual, is(expected));
    }
}
