package ru.yandex.direct.landlord.client

import maps_adv.geosmb.landlord.proto.generate.Generate
import maps_adv.geosmb.landlord.proto.internal.landing_details.LandingDetailsOuterClass.EditLandingDetailsInput
import maps_adv.geosmb.landlord.proto.internal.landing_details.LandingDetailsOuterClass.EditLandingDetailsOutput
import maps_adv.geosmb.landlord.proto.internal.landing_details.LandingDetailsOuterClass.ShowLandingDetailsInput
import maps_adv.geosmb.landlord.proto.internal.landing_details.LandingDetailsOuterClass.ShowLandingDetailsOutput
import maps_adv.geosmb.landlord.proto.organization_details.OrganizationDetailsOuterClass
import org.slf4j.LoggerFactory
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.landlord.client.converter.EditLandingOutputConverter
import ru.yandex.direct.landlord.client.converter.GenerateLandingOutputConverter
import ru.yandex.direct.landlord.client.converter.ShowLandingOutputConverter
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService
import javax.annotation.Nullable

private val logger = LoggerFactory.getLogger(LandlordClient::class.java)

open class LandlordClient(
    private val configuration: LandlordConfiguration,
    fetcherFactory: ParallelFetcherFactory,
    tvmIntegration: TvmIntegration,
) {

    private val api = Smart.builder()
        .withParallelFetcherFactory(fetcherFactory)
        .useTvm(tvmIntegration, TvmService.fromIdStrict(configuration.tvmAppId))
        .withProfileName("landlord_client")
        .withBaseUrl(configuration.apiUrl)
        .withResponseConverterFactory(
            ResponseConverterFactory.builder()
                .addConverters(
                    ShowLandingOutputConverter(),
                    EditLandingOutputConverter(),
                    GenerateLandingOutputConverter()
                )
                .build()
        )
        .build()
        .create(LandlordApi::class.java)

    @Nullable
    fun showLandingDetails(id: Long): ShowLandingDetailsOutput? {
        val input = ShowLandingDetailsInput.newBuilder()
            .setBizId(id)
            .setVersion(OrganizationDetailsOuterClass.LandingVersion.STABLE)
            .build()
        logger.info("Send show landing request: {}", input)
        val call = api.showLandingDetails(input)
        val result = call.execute()
        ErrorUtils.checkResultForErrors(result, { LandlordApiException(it) })
        return result.success
    }

    @Nullable
    fun editLandingDetails(input: EditLandingDetailsInput): EditLandingDetailsOutput? {
        logger.info("Send edit landing request: {}", input)
        val call = api.editLandingDetails(input)
        val result = call.execute()
        ErrorUtils.checkResultForErrors(result, { LandlordApiException(it) })
        return result.success
    }

    @Nullable
    fun generateLandingData(bizId: Long): Generate.GenerateDataOutput? {
        val input = Generate.GenerateDataInput
            .newBuilder()
            .setBizId(bizId)
            .setPublish(true)
            .build()
        logger.info("Send generate landing request: {}", input)
        val call = api.generateLandingData(input)
        val result = call.execute()
        ErrorUtils.checkResultForErrors(result, { LandlordApiException(it) })
        return result.success
    }

    open fun buildUrlFromSlug(slug: String) = "https://$slug${configuration.landingUrlSuffix}"
}
