package ru.yandex.direct.mail;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.annotation.Nullable;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.util.ByteArrayDataSource;

public class MailMessage {
    private final EmailAddress from;
    private EmailAddress to;
    private final String subject;
    private final String messageBody;
    private final EmailContentType contentType;
    private EmailAddress cc;
    private EmailAddress bcc;
    private EmailAddress replyTo;
    private List<MimeBodyPart> attachments;

    public MailMessage(EmailAddress from, EmailAddress to, String subject,
                       String messageBody, EmailContentType contentType, @Nullable List<MimeBodyPart> attachments) {
        this.contentType = contentType == null ? EmailContentType.TEXT : contentType;
        this.subject = subject;
        this.messageBody = messageBody;
        this.from = from;
        this.to = to;
        this.attachments = attachments;
    }

    public MailMessage(EmailAddress from, EmailAddress to, String subject, String messageBody) {
        this(from, to, subject, messageBody, EmailContentType.TEXT, null);
    }

    public EmailAddress getFrom() {
        return from;
    }

    public EmailAddress getTo() {
        return to;
    }

    public String getSubject() {
        return subject;
    }

    public String getMessageBody() {
        return messageBody;
    }

    public EmailContentType getContentType() {
        return contentType;
    }

    public EmailAddress getCc() {
        return cc;
    }

    public EmailAddress getBcc() {
        return bcc;
    }

    public EmailAddress getReplyTo() {
        return replyTo;
    }

    public void withTo(EmailAddress to) {
        this.to = to;
    }

    public void withCc(EmailAddress cc) {
        this.cc = cc;
    }

    public void withBcc(EmailAddress bcc) {
        this.bcc = bcc;
    }

    public void withReplyTo(EmailAddress replyTo) {
        this.replyTo = replyTo;
    }

    public void addAttachment(String contentType, String fileName, byte[] content) {
        DataSource source = new ByteArrayDataSource(content, contentType);
        addAttachment(fileName, source);
    }

    public void addAttachment(String contentType, String fileName, InputStream content) {
        DataSource source;

        try {
            source = new ByteArrayDataSource(content, contentType);
        } catch (IOException e) {
            throw new EmailException(e);
        }

        addAttachment(fileName, source);
    }

    private void addAttachment(String fileName, DataSource source) {
        MimeBodyPart bodyPart = new MimeBodyPart();
        try {
            bodyPart.setDataHandler(new DataHandler(source));
            bodyPart.setFileName(fileName);
        } catch (MessagingException e) {
            throw new EmailException(e);
        }

        synchronized (this) {
            if (attachments == null) {
                attachments = new ArrayList<>();
            }

            attachments.add(bodyPart);
        }
    }

    MimeMessage toMimeMessage(Session mailSession) throws MessagingException {
        MimeMessage result = new MimeMessage(mailSession);
        result.setFrom(from.toInternetAddress());
        result.setRecipient(Message.RecipientType.TO, to.toInternetAddress());
        if (cc != null) {
            result.setRecipient(Message.RecipientType.CC, cc.toInternetAddress());
        }
        if (bcc != null) {
            result.setRecipient(Message.RecipientType.BCC, bcc.toInternetAddress());
        }
        if (replyTo != null) {
            result.setReplyTo(new InternetAddress[]{replyTo.toInternetAddress()});
        }
        result.setSubject(subject, MailUtil.DEFAULT_ENCODING);

        if (attachments == null) {
            result.addHeader("Content-Type", contentType.typeString());
            result.setText(messageBody, MailUtil.DEFAULT_ENCODING, contentType.getSubtype());
        } else {
            Multipart multipart = new MimeMultipart();

            MimeBodyPart textBodyPart = new MimeBodyPart();
            textBodyPart.addHeader("Content-Type", contentType.typeString());
            textBodyPart.setText(messageBody, MailUtil.DEFAULT_ENCODING, contentType.getSubtype());

            multipart.addBodyPart(textBodyPart);

            for (MimeBodyPart attachment : attachments) {
                multipart.addBodyPart(attachment);
            }

            result.setContent(multipart);
        }

        return result;
    }

    public enum EmailContentType {
        TEXT("plain"),
        HTML("html");

        private final String subtype;
        private final String typeString;

        EmailContentType(String subtype) {
            this.subtype = subtype;
            this.typeString = String.format("text/%s; charset=\"%s\"", subtype, MailUtil.DEFAULT_ENCODING);
        }

        public String getSubtype() {
            return subtype;
        }

        public String typeString() {
            return typeString;
        }
    }
}
