package ru.yandex.direct.mail;

import java.io.UnsupportedEncodingException;
import java.util.Properties;

import javax.annotation.Nullable;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.MimeMessage;

import com.google.common.base.Preconditions;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Objects.nonNull;
import static ru.yandex.direct.mail.MailUtil.RFC822_DATE_HEADER;
import static ru.yandex.direct.mail.MailUtil.isNullOrEmpty;

/**
 * Отправляет почту по SMTP-протоколу с использованием JavaMail API
 */
public class SmtpMailSender implements MailSender {
    public static final int DEFAULT_SMTP_PORT = 25;

    private final Session mailSession;
    private final YServiceTokenCreator yServiceTokenCreator;
    private final String serverUrlString;

    /**
     * @param bounceAddress  адрес, на который следует вернуть письмо в случае, если адресат недоступен
     * @param smtpServerPort порт smtp сервера
     * @see com.sun.mail.smtp.SMTPMessage#setEnvelopeFrom(String)
     */
    public SmtpMailSender(String bounceAddress, String smtpServerHost, int smtpServerPort, YServiceTokenCreator tokenCreator) {
        Preconditions.checkNotNull(smtpServerHost);
        Preconditions.checkArgument(smtpServerPort > 0);
        Properties mailSessionProperties = createSessionProperties(bounceAddress, smtpServerHost, smtpServerPort);
        mailSession = Session.getInstance(mailSessionProperties);
        yServiceTokenCreator = tokenCreator;
        serverUrlString = String.format("smtp://%s:%d", smtpServerHost, smtpServerPort);
    }

    /**
     * @param bounceAddress адрес, на который следует вернуть письмо в случае, если адресат недоступен
     */
    public SmtpMailSender(String bounceAddress, YServiceTokenCreator tokenCreator) {
        this(bounceAddress, "localhost", DEFAULT_SMTP_PORT, tokenCreator);
    }

    /**
     * Подготавливает переменные среды для JavaMail.
     * Выставить {@code mail.mime.charset = "UTF-8"} через {@link Properties} не получится, т.к.
     * {@link javax.mail.internet.MimeUtility#getDefaultJavaCharset}
     * использует {@link System#getProperty(String)}. Вместо этого приходится использовать {@link MailUtil#DEFAULT_ENCODING} то здесь, то там в коде.
     * Без этого JavaMail будет использовать кодировку по-умолчанию, из-за этого тесты могут вести себя нестабильно.
     *
     * @param bounceAddress
     * @param smtpServerPort
     * @return
     * @see com.sun.mail.smtp.SMTPMessage#setEnvelopeFrom(String)
     */
    private Properties createSessionProperties(String bounceAddress, String smtpHost, int smtpServerPort) {
        Properties mailSessionProperties = new Properties();
        mailSessionProperties.setProperty("mail.transport.protocol", "smtp");
        mailSessionProperties.setProperty("mail.smtp.host", smtpHost);
        mailSessionProperties.setProperty("mail.smtp.port", Integer.toString(smtpServerPort));
        //разрешаем использовать 'Content-Transfer-Encoding':'8bit' в случае, если сервер поддерживает расширение SMTP протокола '8BITMIME'
        mailSessionProperties.setProperty("mail.smtp.allow8bitmime", "true");
        if (bounceAddress != null) {
            //Добавить Bounce Address (см. https://en.wikipedia.org/wiki/Bounce_address);
            //у Unix sendmail за это поведение отвечает параметр командной строки '-f'
            //если захотим каким-то(в пределах запуска приложения) письмам проставлять заголовок, а каким-то нет - придется
            //создавать SMTP сообщения напрямую и вызывать com.sun.mail.smtp.SMTPMessage#setEnvelopeFrom(String)
            mailSessionProperties.setProperty("mail.smtp.from", bounceAddress);
        }
        return mailSessionProperties;
    }

    @Override
    public void send(AuthorizedMailMessage message) {
        MimeMessage mimeMessage = getMimeMessage(message);

        tryAddXYandexDirectHeaders(mimeMessage,
                message.getOperatorUid(), message.getSubjectUserClientId(), message.getSubject());

        sendMimeMessage(mimeMessage);
    }

    @Override
    public void send(MailMessage message) {
        MimeMessage mimeMessage = getMimeMessage(message);
        sendMimeMessage(mimeMessage);
    }

    private void tryAddXYandexDirectHeaders(MimeMessage mimeMessage,
                                            @Nullable Long operatorUid, @Nullable Long clientId, @Nullable String subject) {
        String hmacToken = yServiceTokenCreator.createHmacToken(operatorUid, clientId, subject);

        if (nonNull(hmacToken)) {
            try {
                mimeMessage.setHeader(YServiceTokenCreator.X_CLIENT_ID_HEADER, clientId.toString());
                mimeMessage.setHeader(YServiceTokenCreator.X_OPERATOR_UID_HEADER, operatorUid.toString());
                mimeMessage.setHeader(YServiceTokenCreator.X_HMAC_SIGN_HEADER, hmacToken);
            } catch (MessagingException ex) {
                throw new EmailException(ex);
            }
        }
    }


    private MimeMessage getMimeMessage(MailMessage message) {
        try {
            return message.toMimeMessage(mailSession);
        } catch (MessagingException ex) {
            throw new EmailException(ex);
        }
    }

    private void sendMimeMessage(MimeMessage mimeMessage) {
        try {
            //нестандартный заголовок Precedence: указывает, что письмо - часть рассылки
            mimeMessage.addHeader("Precedence", "bulk");
            YServiceToken token = yServiceTokenCreator.createToken(mimeMessage);
            if (token != null) {
                if (isNullOrEmpty(mimeMessage.getHeader(RFC822_DATE_HEADER))) {
                    mimeMessage.setHeader(RFC822_DATE_HEADER, token.getDate());
                }
                mimeMessage.setHeader(YServiceTokenCreator.X_SERVICE_HEADER, token.getTokenValue());
            }
            try (TraceProfile ignore = Trace.current().profile("mail:send", serverUrlString)) {
                Transport.send(mimeMessage);
            }
        } catch (MessagingException | UnsupportedEncodingException ex) {
            throw new EmailException(ex);
        }
    }
}
