package ru.yandex.direct.mail;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.List;
import java.util.Locale;
import java.util.StringJoiner;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MailDateFormat;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.HashingUtils;

import static java.util.Arrays.asList;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.mail.MailUtil.RFC822_DATE_HEADER;
import static ru.yandex.direct.mail.MailUtil.isNullOrEmpty;

/**
 * Добавляет служебные заголовки письмам.
 * {@value X_SERVICE_HEADER}.
 *
 * @see <a href="https://wiki.yandex-team.ru/EgorGanin/PrometkiizSO/#kodirovaniezagolovkax-yandex-service">Кодирование заголовка X-Yandex-Service</a>
 * <p>
 * {@value X_HMAC_SIGN_HEADER, @value X_CLIENT_ID_HEADER, @value X_OPERATOR_UID_HEADER}
 * @see <a href="https://st.yandex-team.ru/DIRECT-118801">Добавлять к письмам, отправляемым в CRM заголовки с данными целевого клиента и оператора</a>
 */

public class YServiceTokenCreator {
    public static final String X_SERVICE_HEADER = "X-Yandex-Service";
    public static final String X_CLIENT_ID_HEADER = "X-Yandex-Direct-Client-Id";
    public static final String X_OPERATOR_UID_HEADER = "X-Yandex-Direct-Operator-Uid";
    public static final String X_HMAC_SIGN_HEADER = "X-Yandex-Direct-Sign";

    private static final Logger logger = LoggerFactory.getLogger(YServiceTokenCreator.class);
    private static final List<String>
            YANDEX_TLD_LIST =
            asList("ya.ru", "yandex.ru", "yandex.com", "narod.ru", "yandex.ua", "yandex.kz", "yandex.by");
    private static final DateTimeFormatter
            RFC_2822_FORMAT = DateTimeFormatter.ofPattern(new MailDateFormat().toPattern(), Locale.US);
    private static final String HMAC_MD5 = "HmacMD5";

    private final String serviceName;
    private final String serviceSalt;
    private final String hmacSalt;

    public YServiceTokenCreator(String serviceName, String serviceSalt, String hmacSalt) {
        this.serviceName = serviceName;
        this.serviceSalt = serviceSalt;
        this.hmacSalt = hmacSalt;
    }

    public YServiceToken createToken(MimeMessage message) throws MessagingException, UnsupportedEncodingException {
        String to = ((InternetAddress) message.getRecipients(Message.RecipientType.TO)[0]).getAddress();
        Address[] bcc = message.getRecipients(Message.RecipientType.BCC);
        Address[] cc = message.getRecipients(Message.RecipientType.CC);
        if (isNullOrEmpty(bcc) && isNullOrEmpty(cc) && isYandexDestination(to)) {
            String date = getDateForMessage(message);
            String from = MimeUtility.decodeText(InternetAddress.toString(message.getFrom()));
            StringJoiner joiner = new StringJoiner("_");
            joiner.add(date).add(from).add(message.getSubject()).add(serviceName).add(serviceSalt);
            String saltedString = joiner.toString();

            String hexed = md5Hex(saltedString);
            String rawToken = serviceName + " " + hexed;
            String encodedToken =
                    new String(Base64.getMimeEncoder().encode(rawToken.getBytes(StandardCharsets.US_ASCII)));
            return new YServiceToken(date, encodedToken);
        }
        return null;
    }

    @Nullable
    public String createHmacToken(@Nullable Long operatorUid, @Nullable Long clientId, @Nullable String subject) {
        if (isNull(operatorUid) || isNull(clientId)) {
            return null;
        }

        return hmacMD5(hmacSalt, operatorUid.toString(), clientId.toString(), defaultIfNull(subject, ""));
    }

    private String md5Hex(String src) {
        return HashingUtils.getMd5HashAsHexString(src.getBytes(StandardCharsets.UTF_8));
    }

    private String getDateForMessage(MimeMessage message) throws MessagingException {
        String[] dateHeader = message.getHeader(RFC822_DATE_HEADER);
        String date;
        if (dateHeader != null) {
            if (dateHeader.length == 1) {
                date = dateHeader[0];
            } else {
                date = RFC_2822_FORMAT.format(ZonedDateTime.now());
                logger.warn("date header has zero or multiple values");
            }
        } else {
            date = RFC_2822_FORMAT.format(ZonedDateTime.now());
        }
        return date;
    }

    private boolean isYandexDestination(String dest) {
        for (String tld : YANDEX_TLD_LIST) {
            if (dest.endsWith(tld)) {
                return true;
            }
        }
        return false;
    }

    //На базе https://gist.github.com/MaximeFrancoeur/bcb7fc2db08c704f322a
    private String hmacMD5(String keyString, String... items) {
        String digest = null;
        try {
            SecretKeySpec key = new SecretKeySpec((keyString).getBytes(StandardCharsets.US_ASCII), HMAC_MD5);
            Mac mac = Mac.getInstance(HMAC_MD5);
            mac.init(key);

            String joinedItems = joinStrings(items, "");
            byte[] bytes = mac.doFinal(joinedItems.getBytes(StandardCharsets.UTF_8));

            digest = bytesToHexString(bytes);

        } catch (InvalidKeyException | NoSuchAlgorithmException e) {
            logger.warn("Error while hmac signature generation for "
                    + joinStrings(items, ", ") + ". Got exception :" + e.toString(), e);
        }
        return digest;
    }

    private String joinStrings(String[] strings, String delimiter) {
        return asList(strings).stream().collect(Collectors.joining(delimiter));
    }

    private String bytesToHexString(byte[] bytes) {
        StringBuffer hash = new StringBuffer();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(0xFF & bytes[i]);
            if (hex.length() == 1) {
                hash.append('0');
            }
            hash.append(hex);
        }
        return hash.toString();
    }
}
