package ru.yandex.direct.mediascope;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HttpHeaders;
import io.netty.handler.codec.http.HttpMethod;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Param;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mediascope.model.request.MediascopePosition;
import ru.yandex.direct.mediascope.model.request.MediascopeTokensRequest;
import ru.yandex.direct.mediascope.model.response.MediascopePositionsResponse;
import ru.yandex.direct.mediascope.model.response.MediascopePrefixResponse;
import ru.yandex.direct.mediascope.model.response.MediascopeTokensResponse;
import ru.yandex.direct.mediascope.model.response.MediascopeUserInfoResponse;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.tracing.Trace;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNPARSABLE;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@ParametersAreNonnullByDefault
public class MediascopeClient {
    private static final Logger logger = LoggerFactory.getLogger(MediascopeClient.class);
    private static final int REQUEST_TIMEOUT = 10_000;
    private static final String TRACE_FUNC = "mediascope_integration:api";
    private static final String SEND_POSITIONS = "send_positions";
    private static final String GET_NEW_TOKENS = "get_new_tokens";
    private static final String GET_FRESH_TOKENS = "get_fresh_tokens";
    private static final String GET_PREFIX = "get_prefix";
    private static final String GET_USER_INFO = "get_user_info";
    private static final Pattern POSITION_ID_PATTERN = Pattern.compile("'([\\w-]*)'");

    private static final String EXTERNAL_SYSTEM = "mediascope_integration";
    private static final SolomonExternalSystemMonitorService MONITOR_SERVICE = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(SEND_POSITIONS, GET_NEW_TOKENS, GET_FRESH_TOKENS, GET_PREFIX, GET_USER_INFO)
    );

    private final AsyncHttpClient asyncHttpClient;
    private final MediascopeClientConfig mediascopeClientConfig;

    public MediascopeClient(
            AsyncHttpClient asyncHttpClient,
            MediascopeClientConfig mediascopeClientConfig
    ) {
        this.asyncHttpClient = asyncHttpClient;
        this.mediascopeClientConfig = mediascopeClientConfig;
    }

    public MediascopePositionsResponse sendPositions(String accessToken, List<MediascopePosition> positions) {
        var request = new RequestBuilder()
                .setUrl(mediascopeClientConfig.getApiUrl() + "/positions")
                .setMethod(HttpMethod.POST.name())
                .setHeader(HttpHeaders.CONTENT_TYPE, "application/json")
                .setHeader(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken)
                .setBody(toJson(positions))
                .build();

        var response = sendRequest(request, SEND_POSITIONS, true);
        var errors = listToMap(positions, MediascopePosition::getPositionId, p -> "Server error");
        var defaultErrorResponse = new MediascopePositionsResponse(500L, "Error", "", errors);

        if (response == null) {
            MONITOR_SERVICE.write(SEND_POSITIONS, STATUS_5XX);
            return defaultErrorResponse;
        }

        if (response.getStatusCode() != 200) {
            MONITOR_SERVICE.write(SEND_POSITIONS, STATUS_4XX);
            MediascopePositionsResponse triedParsedResponse = parsePositionsResponse(response);
            if (triedParsedResponse == null) {
                MONITOR_SERVICE.write(SEND_POSITIONS, STATUS_UNPARSABLE);
                return defaultErrorResponse;
            } else {
                return triedParsedResponse;
            }
        } else {
            MONITOR_SERVICE.write(SEND_POSITIONS, STATUS_2XX);
            return new MediascopePositionsResponse(200L, "Ok", "", emptyMap());
        }
    }

    public MediascopeTokensResponse getNewTokens(MediascopeTokensRequest tokensRequest) {
        var request = new RequestBuilder()
                .setUrl(mediascopeClientConfig.getOidcUrl() + "/token")
                .setMethod(HttpMethod.POST.name())
                .setHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded")
                .setFormParams(List.of(
                        new Param("client_id", mediascopeClientConfig.getClientId()),
                        new Param("client_secret", mediascopeClientConfig.getClientSecret()),
                        new Param("grant_type", "authorization_code"),
                        new Param("code", tokensRequest.getCode()),
                        new Param("code_verifier", tokensRequest.getCodeVerifier()),
                        new Param("redirect_uri", tokensRequest.getRedirectUri()),
                        new Param("scope", "openid profile offline_access")
                ))
                .setRequestTimeout(REQUEST_TIMEOUT)
                .build();

        return sendRequestAndParseResponse(request, GET_NEW_TOKENS, MediascopeTokensResponse.class);
    }

    public MediascopeTokensResponse getFreshTokens(String refreshToken) {
        var request = new RequestBuilder()
                .setUrl(mediascopeClientConfig.getOidcUrl() + "/token")
                .setMethod(HttpMethod.POST.name())
                .setHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded")
                .setFormParams(List.of(
                        new Param("client_id", mediascopeClientConfig.getClientId()),
                        new Param("client_secret", mediascopeClientConfig.getClientSecret()),
                        new Param("grant_type", "refresh_token"),
                        new Param("refresh_token", refreshToken),
                        new Param("scope", "openid profile offline_access")
                ))
                .setRequestTimeout(REQUEST_TIMEOUT)
                .build();

        return sendRequestAndParseResponse(request, GET_FRESH_TOKENS, MediascopeTokensResponse.class);
    }

    public MediascopePrefixResponse getPrefix(String accessToken) {
        var request = new RequestBuilder()
                .setUrl(mediascopeClientConfig.getApiUrl() + "/tmsecprefix")
                .setMethod(HttpMethod.GET.name())
                .setHeader(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken)
                .setRequestTimeout(REQUEST_TIMEOUT)
                .build();

        return sendRequestAndParseResponse(request, GET_PREFIX, MediascopePrefixResponse.class);
    }

    public MediascopeUserInfoResponse getUserInfo(String accessToken) {
        var request = new RequestBuilder()
                .setUrl(mediascopeClientConfig.getOidcUrl() + "/userinfo")
                .setMethod(HttpMethod.GET.name())
                .setHeader(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken)
                .setRequestTimeout(REQUEST_TIMEOUT)
                .build();
        return sendRequestAndParseResponse(request, GET_USER_INFO, MediascopeUserInfoResponse.class);
    }

    @Nullable
    private <T> T sendRequestAndParseResponse(Request request, String methodName, Class<T> type) {
        var response = sendRequest(request, methodName, false);
        if (response == null) {
            MONITOR_SERVICE.write(methodName, STATUS_5XX);
            return null;
        }
        if (response.getStatusCode() != 200) {
            MONITOR_SERVICE.write(methodName, STATUS_4XX);
            return null;
        }
        try {
            var parsedResponse = fromJson(response.getResponseBodyAsStream(), type);
            MONITOR_SERVICE.write(methodName, STATUS_2XX);
            return parsedResponse;
        } catch (IllegalArgumentException e) {
            MONITOR_SERVICE.write(methodName, STATUS_UNPARSABLE);
            throw e;
        }
    }

    private Response sendRequest(Request request, String methodName, boolean logParams) {
        try (var ignore = Trace.current().profile(TRACE_FUNC, methodName)) {
            try {
                var response = asyncHttpClient.executeRequest(request).get();
                logRequest(methodName, request, response, null, logParams);
                return response;
            } catch (InterruptedException e) {
                logRequest(methodName, request, null, e, false);
                Thread.currentThread().interrupt();
            } catch (RuntimeException | ExecutionException e) {
                logRequest(methodName, request, null, e, false);
            }
        }
        return null;
    }

    private void logRequest(
            String methodName,
            Request request,
            @Nullable Response response,
            @Nullable Exception ex,
            boolean logParams
    ) {
        StringBuilder message = new StringBuilder()
                .append("Sending request ").append(methodName).append(" to Mediascope");

        if (logParams) {
            message.append("\n").append("Request: ").append(request.getStringData());
        }

        if (response != null && (logParams || response.getStatusCode() != 200)) {
            message.append("\n").append("Response: ").append(getLogMessage(response));
        }

        if (ex != null) {
            message.append("\n").append("Exception: ").append(ex.toString());
        }

        if ((response != null && response.getStatusCode() != 200) || ex != null) {
            logger.error(message.toString());
        } else {
            logger.info(message.toString());
        }
    }

    private String getLogMessage(Response response) {
        StringBuilder builder = new StringBuilder()
                .append(response.getStatusCode())
                .append("\n");

        for (Map.Entry<String, String> entry : response.getHeaders()) {
            builder.append(entry.getKey())
                    .append("=")
                    .append(entry.getValue())
                    .append("\n");
        }

        return builder.append(new String(response.getResponseBodyAsBytes())).toString();
    }

    private MediascopePositionsResponse parsePositionsResponse(Response response) {
        try {
            var result = fromJson(response.getResponseBodyAsStream(), MediascopePositionsResponse.class);

            var positionToErrorMap = new HashMap<String, String>();
            for (var key : result.getErrors().keySet()) {
                try {
                    var matcher = POSITION_ID_PATTERN.matcher(key);
                    if (matcher.find()) {
                        var positionId = matcher.group(1);
                        if (positionId != null) {
                            positionToErrorMap.put(positionId, result.getErrors().get(key));
                        }
                    }
                } catch (IllegalStateException ex) {
                    logger.warn("Unable to parse positionId from " + key, ex);
                }
            }

            result.setErrors(positionToErrorMap);
            return result;
        } catch (IllegalArgumentException ex) {
            logger.error("Couldn't parse Mediascope response to send positions request", ex);
            return null;
        }
    }
}
