package ru.yandex.direct.metric.collector;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Инициализация регулярного обхода MerticProvider-ов
 */
@ParametersAreNonnullByDefault
public class MetricCollector implements AutoCloseable {
    private static final Duration DEFAULT_INTERVAL = Duration.ofSeconds(60);
    private static final Logger logger = LoggerFactory.getLogger(MetricCollector.class);
    private static final int DEFAULT_POOL_SIZE = 1;

    private final Collection<MetricProvider> metricProviders;
    private final ScheduledExecutorService executor;
    private final boolean needToShutdownExecutor;

    private boolean inited = false;

    /**
     * @param metricProviders     - набор MetricProvider-ов, из которых нужно собирать данные
     * @param executor            - пул для запуска провайдеров
     */
    public MetricCollector(
            Collection<MetricProvider> metricProviders,
            @Nullable ScheduledExecutorService executor
    ) {
        this.metricProviders = new ArrayList<>(metricProviders);
        if (executor == null) {
            this.executor = createDefaultExecutor();
            needToShutdownExecutor = true;
        } else {
            this.executor = executor;
            needToShutdownExecutor = false;
        }
    }

    private ScheduledExecutorService createDefaultExecutor() {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("metric-collector-%02d").build();
        return Executors.newScheduledThreadPool(DEFAULT_POOL_SIZE, threadFactory);
    }

    /**
     * Планирование запусков задач
     */
    public void init() {
        logger.info("init");
        if (!inited) {
            internalInit();
            inited = true;
        }
    }

    private void internalInit() {
        for (MetricProvider provider : metricProviders) {
            executor.scheduleAtFixedRate(
                    () -> processProvider(provider),
                    0, DEFAULT_INTERVAL.toMillis(), TimeUnit.MILLISECONDS
            );
        }
    }

    private void processProvider(MetricProvider provider) {
        logger.trace("run metric provider without graphite {}", provider.getClass().getCanonicalName());
        try {
            provider.provideMetrics(null);
        } catch (RuntimeException e) {
            logger.error("Exception", e);
        }
    }

    @Override
    public void close() {
        logger.info("close");
        if (needToShutdownExecutor && !executor.isShutdown()) {
            executor.shutdown();
        }
    }
}
