package ru.yandex.direct.metrika.client.asynchttp;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.time.Period;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Joiner;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.http.HttpStatus;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.MetrikaConfiguration;
import ru.yandex.direct.metrika.client.UpdateCounterGrantsMetrikaRequest;
import ru.yandex.direct.metrika.client.UpdateCounterGrantsMetrikaRequestCounter;
import ru.yandex.direct.metrika.client.UpdateCounterGrantsMetrikaRequestGrant;
import ru.yandex.direct.metrika.client.internal.CountersGoalsJsonResponseConverter;
import ru.yandex.direct.metrika.client.internal.CreateGoalRequestEnvelope;
import ru.yandex.direct.metrika.client.internal.CreateSegmentRequestEnvelope;
import ru.yandex.direct.metrika.client.internal.Dimension;
import ru.yandex.direct.metrika.client.internal.EstimateUsersByConditionEnvelope;
import ru.yandex.direct.metrika.client.internal.GetCounterResponseEnvelope;
import ru.yandex.direct.metrika.client.internal.GetStatByDateResult;
import ru.yandex.direct.metrika.client.internal.GetStatResult;
import ru.yandex.direct.metrika.client.internal.MetrikaByTimeStatisticsParams;
import ru.yandex.direct.metrika.client.internal.MetrikaConditionBuilder;
import ru.yandex.direct.metrika.client.internal.MetrikaSourcesParams;
import ru.yandex.direct.metrika.client.internal.StatData;
import ru.yandex.direct.metrika.client.model.request.GetExistentCountersRequest;
import ru.yandex.direct.metrika.client.model.request.GetGoalsRequest;
import ru.yandex.direct.metrika.client.model.request.GrantAccessRequestStatusesRequest;
import ru.yandex.direct.metrika.client.model.request.RequestGrantsRequest;
import ru.yandex.direct.metrika.client.model.request.RetargetingGoalGroup;
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter;
import ru.yandex.direct.metrika.client.model.response.Counter;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CounterGoalExtended;
import ru.yandex.direct.metrika.client.model.response.CreateCounterGoal;
import ru.yandex.direct.metrika.client.model.response.GetCountersGoalsResponse;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponse;
import ru.yandex.direct.metrika.client.model.response.GetGoalsResponse;
import ru.yandex.direct.metrika.client.model.response.GoalConversionInfo;
import ru.yandex.direct.metrika.client.model.response.GoalConversionsCountResponse;
import ru.yandex.direct.metrika.client.model.response.GrantAccessRequestStatusesResponse;
import ru.yandex.direct.metrika.client.model.response.MetrikaErrorResponse;
import ru.yandex.direct.metrika.client.model.response.RequestGrantsResponse;
import ru.yandex.direct.metrika.client.model.response.RetargetingCondition;
import ru.yandex.direct.metrika.client.model.response.Segment;
import ru.yandex.direct.metrika.client.model.response.TurnOnCallTrackingResponse;
import ru.yandex.direct.metrika.client.model.response.UpdateCounterGrantsResponse;
import ru.yandex.direct.metrika.client.model.response.UserCounters;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtendedResponse;
import ru.yandex.direct.metrika.client.model.response.UserCountersResponse;
import ru.yandex.direct.metrika.client.model.response.sources.SourcesResponse;
import ru.yandex.direct.metrika.client.model.response.sources.TrafficSource;
import ru.yandex.direct.metrika.client.model.response.statistics.StatisticsResponse;
import ru.yandex.direct.metrika.client.model.response.statistics.StatisticsResponseRow;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmIntegrationStub;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.inside.passport.tvm2.TvmHeaders;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyMap;
import static org.apache.commons.lang3.StringUtils.wrap;
import static ru.yandex.direct.asynchttp.FetcherSettings.DEFAULT_REQUEST_TIMEOUT;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;


public class MetrikaAsyncHttpClient implements MetrikaClient {
    private static final Logger logger = LoggerFactory.getLogger(MetrikaAsyncHttpClient.class);

    private static final DateTimeFormatter METRIKA_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final Integer DEFAULT_TIMEOUT_SECONDS = DEFAULT_REQUEST_TIMEOUT.toSecondsPart();
    public static final int MAX_COUNTERS_SIZE = 200;
    public static final int GOALS_STATISTIC_LIMIT = 100_000;

    private final String intapiUrl;
    private final String audienceUrl;
    private final String metrikaUrl;
    private IntApi intApi;
    private AudienceApi audienceApi;
    private MetrikaApi metrikaApi;
    private final PpcProperty<Integer> parallelRequestsProperty;
    private final PpcProperty<Integer> maxCountersInProductImpressionsProperty;

    private final PpcProperty<Integer> metrikaIntapiTimeoutDurationSeconds;
    private final PpcProperty<Integer> metrikaAudienceApiTimeoutDurationSeconds;
    private final PpcProperty<Integer> metrikaApiTimeoutDurationSeconds;

    public MetrikaAsyncHttpClient(PpcPropertiesSupport ppcPropertiesSupport) {
        this(new MetrikaConfiguration(),
                new DefaultAsyncHttpClient(),
                new TvmIntegrationStub(), ppcPropertiesSupport, false);
    }

    public MetrikaAsyncHttpClient(MetrikaConfiguration configuration, AsyncHttpClient asyncHttpClient,
                                  TvmIntegration tvmIntegration, PpcPropertiesSupport ppcPropertiesSupport,
                                  boolean isProd) {

        intapiUrl = configuration.getIntapiUrl();
        audienceUrl = configuration.getAudienceUrl();
        metrikaUrl = configuration.getMetrikaUrl();
        this.parallelRequestsProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.NUMBER_OF_PARALLEL_REQUESTS_TO_METRIKA,
                Duration.ofMinutes(1));
        this.maxCountersInProductImpressionsProperty = ppcPropertiesSupport.get(
                PpcPropertyNames.MAX_COUNTERS_IN_PRODUCT_IMPRESSIONS,
                Duration.ofMinutes(1));

        this.metrikaIntapiTimeoutDurationSeconds = ppcPropertiesSupport.get(
                PpcPropertyNames.METRIKA_INTAPI_TIMEOUT_DURATION_SECONDS,
                Duration.ofMinutes(1));
        this.metrikaAudienceApiTimeoutDurationSeconds = ppcPropertiesSupport.get(
                PpcPropertyNames.METRIKA_AUDIENCE_API_TIMEOUT_DURATION_SECONDS,
                Duration.ofMinutes(1));
        this.metrikaApiTimeoutDurationSeconds = ppcPropertiesSupport.get(
                PpcPropertyNames.METRIKA_API_TIMEOUT_DURATION_SECONDS,
                Duration.ofMinutes(1));

        initApi(asyncHttpClient, tvmIntegration, isProd);
    }

    private void initApi(AsyncHttpClient asyncHttpClient, TvmIntegration tvmIntegration, boolean isProd) {

        var factory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        intApi = Smart.builder()
                .withParallelFetcherFactory(factory)
                .useTvm(tvmIntegration,
                        isProd ? TvmService.METRIKA_INTERNAL_API_PROD : TvmService.METRIKA_INTERNAL_API_TEST)
                .withProfileName("metrika_int_api")
                .withBaseUrl(intapiUrl)
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"))
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(new CountersGoalsJsonResponseConverter())
                        .build())
                .build()
                .create(IntApi.class);

        audienceApi = Smart.builder()
                .withParallelFetcherFactory(factory)
                .useTvm(tvmIntegration,
                        isProd ? TvmService.METRIKA_AUDIENCE_PROD
                                : TvmService.METRIKA_AUDIENCE_TEST)
                .withProfileName("metrika_audience_api")
                .withBaseUrl(audienceUrl)
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"))
                .build()
                .create(AudienceApi.class);

        metrikaApi = Smart.builder()
                .withParallelFetcherFactory(factory)
                .useTvm(tvmIntegration,
                        isProd ? TvmService.METRIKA_API_PROD : TvmService.METRIKA_API_TEST)
                .withProfileName("metrika_public_api")
                .withBaseUrl(metrikaUrl)
                .addHeaderConfigurator(headers -> headers.add("Content-type", "application/json"))
                .build()
                .create(MetrikaApi.class);
    }

    private FetcherSettings getFetcherSettings(PpcProperty<Integer> ppcProperty) {
        var requestTimeout = ppcProperty == null ? DEFAULT_TIMEOUT_SECONDS :
                ppcProperty.getOrDefault(DEFAULT_TIMEOUT_SECONDS);
        return new FetcherSettings()
                .withRequestTimeout(Duration.ofSeconds(requestTimeout));
    }

    /**
     * Can be overridden in tests
     */
    protected LocalDate getLocalDate() {
        return LocalDate.now();
    }

    @Override
    public List<UserCounters> getUsersCountersNum(List<Long> uids) {
        String uidsArg = Joiner.on(',').join(uids);
        Result<List<UserCounters>> result = intApi
                .getUsersCountersNum(uidsArg)
                .execute(getFetcherSettings(metrikaIntapiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, uidsArg);
        return result.getSuccess();
    }

    @Override
    public UserCountersResponse getUsersCountersNum2(List<Long> uids, Collection<Long> counterIds) {
        String uidsArg = Joiner.on(',').join(uids);
        String counterIdsArg = Joiner.on(',').join(counterIds);
        Result<UserCountersResponse> result = intApi
                .getUsersCountersNum2(uidsArg, counterIdsArg)
                .execute(getFetcherSettings(metrikaIntapiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, uidsArg);
        return result.getSuccess();
    }

    @Override
    public List<UserCountersExtended> getUsersCountersNumExtended(List<Long> uids) {
        String uidsArg = Joiner.on(',').join(uids);
        Result<List<UserCountersExtended>> result = intApi
                .getUsersCountersNumExtended(uidsArg)
                .execute(getFetcherSettings(metrikaIntapiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, uidsArg);
        return result.getSuccess();
    }

    @Override
    public UserCountersExtendedResponse getUsersCountersNumExtended2(List<Long> uids,
                                                                     UserCountersExtendedFilter filter) {
        String uidsArg = Joiner.on(',').join(uids);
        String counterIds = filter.getCounterIds() == null || filter.getCounterIds().isEmpty() ? null
                : Joiner.on(',').join(filter.getCounterIds());
        String prefix = filter.getPrefix() == null || filter.getPrefix().isEmpty() ? null : filter.getPrefix();
        Result<UserCountersExtendedResponse> result = intApi
                .getUsersCountersNumExtended2(uidsArg, counterIds, prefix, filter.getLimit())
                .execute(getFetcherSettings(metrikaIntapiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, uidsArg);
        return result.getSuccess();
    }

    public Map<Long, List<RetargetingCondition>> getGoalsByUids(Collection<Long> uids) {
        String uid = "[" + Joiner.on(',').join(uids) + "]";
        Result<Map<Long, List<RetargetingCondition>>> result = audienceApi
                .getGoalsByUids(uid)
                .execute(getFetcherSettings(metrikaAudienceApiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, uid);
        return result.getSuccess();
    }

    @Override
    public GetGoalsResponse getGoals(GetGoalsRequest request) {
        String uids = "[" + Joiner.on(',').join(request.getUids()) + "]";
        String ids = request.getIds() == null ? null : "[" + Joiner.on(',').join(request.getIds()) + "]";
        Result<GetGoalsResponse> result = audienceApi
                .getGoals(uids, request.getGoalType(), request.getPrefix(), ids, request.getCountCountersAsGoals())
                .execute(getFetcherSettings(metrikaAudienceApiTimeoutDurationSeconds));
        checkResultForErrors(result, MetrikaClientException::new, request);
        return result.getSuccess();
    }

    @Override
    public long estimateUsersByCondition(List<RetargetingGoalGroup> condition) {
        MetrikaConditionBuilder builder = new MetrikaConditionBuilder();
        builder.appendCondition(condition);
        Result<EstimateUsersByConditionEnvelope> result = audienceApi.estimateUsersByCondition(builder)
                .execute(getFetcherSettings(metrikaAudienceApiTimeoutDurationSeconds));

        checkResultForErrors(result, MetrikaClientException::new, builder);
        return result.getSuccess().getResponse();
    }

    @Override
    public Map<Long, Long> getUserNumberByRegions(Set<Long> counterIds, int limit, int days) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        LocalDate now = getLocalDate();
        String date1 = METRIKA_DATE_FORMAT.format(now.minus(Period.ofDays(days)));
        String date2 = METRIKA_DATE_FORMAT.format(now);
        Set<String> dimensions = Set.of("ym:s:regionCountry", "ym:s:regionArea", "ym:s:regionCity");
        Collection<Result<GetStatResult>> results = metrikaApi
                .getUserCountByDimensions(counterIds, dimensions, limit, date1, date2)
                .execute(1)
                .values();

        Map<Long, Long> regionToUserCount = StreamEx.of(results)
                .map(Result::getSuccess)
                .nonNull()
                .flatCollection(GetStatResult::getData)
                .mapToEntry(StatData::getDimensions, StatData::getMetricAsLong)
                .nonNullKeys()
                .filterKeys(dims -> !dims.isEmpty())
                .mapKeys(dims -> dims.iterator().next())
                .nonNullKeys()
                .mapKeys(Dimension::getId)
                .mapKeys(Long::valueOf)
                .toMap();

        if (regionToUserCount.isEmpty()) {
            results.forEach(result -> checkResultForErrors(
                    result,
                    MetrikaClientException::new,
                    dimensions, counterIds, limit, date1, date2));
        }
        return regionToUserCount;
    }

    @Override
    public Map<Long, Double> getProductImpressionsByCounterId(Set<Long> counterIds, int days) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        LocalDate now = getLocalDate();
        String date1 = METRIKA_DATE_FORMAT.format(now.minus(Period.ofDays(days)));
        String date2 = METRIKA_DATE_FORMAT.format(now);
        Collection<Result<GetStatResult>> results = metrikaApi
                .getProductImpressions(counterIds, date1, date2)
                .execute(maxCountersInProductImpressionsProperty.getOrDefault(10),
                        parallelRequestsProperty.getOrDefault(1)).values();
        results.forEach(result -> processProductImpressionResult(result, counterIds, date1, date2));

        List<StatData> productImpressionStatData = StreamEx.of(results)
                .map(Result::getSuccess)
                .nonNull()
                .toFlatList(GetStatResult::getData);
        return listToMap(productImpressionStatData, StatData::getFirstDimensionIdAsLong, StatData::getMetricAsDouble);
    }

    /**
     * Не бросаем исключение если от метрики пришел 400 ответ -
     * такое может быть когда сделали запрос для одного счетчика без ecommerce.
     */
    private void processProductImpressionResult(Result<GetStatResult> result, Set<Long> counterIds,
                                                String date1, String date2) {
        if (result.getErrors() != null && result.getErrors().size() == 1) {
            Throwable throwable = result.getErrors().get(0);
            if (throwable instanceof ErrorResponseWrapperException) {
                var responseException = (ErrorResponseWrapperException) throwable;
                if (responseException.getResponse() != null &&
                        responseException.getResponse().getStatusCode() == HttpStatus.SC_BAD_REQUEST) {
                    return;
                }
            }
        }
        checkResultForErrors(result, MetrikaClientException::new, counterIds, date1, date2);
    }

    @Override
    public SourcesResponse getAvailableSources(MetrikaSourcesParams params) {
        var attribution = params.getAttribution();
        var attributionStr = attribution == null ? null : attribution.name();

        var trafficSource = "ym:s:<attribution>TrafficSource";
        var sourceEngine = "ym:s:<attribution>SourceEngine";
        var dimensions = String.join(",", trafficSource, sourceEngine);

        var filterList = new ArrayList<String>();
        if (isNotEmpty(params.getCategories())) {
            filterList.add(buildDimensionInListFilter(trafficSource, params.getCategories()));
        }
        if (isNotEmpty(params.getChannels())) {
            filterList.add(buildDimensionInListFilter(sourceEngine, params.getChannels()));
        }
        String filters = nullIfBlank(String.join(" OR ", filterList));

        var rawResult = metrikaApi.getStatData(
                params.getCounterId(),
                attributionStr,
                dimensions,
                "ym:s:visits",
                filters,
                params.getLimit(),
                params.getDateFrom().toString(),
                params.getDateTo().toString()
        ).execute();

        checkResultForErrors(rawResult, MetrikaClientException::new);

        var result = rawResult.getSuccess();

        var sources = StreamEx.of(result.getData())
                .map(StatData::getDimensions)
                .map(dims -> new TrafficSource(dims.get(0), dims.get(1)))
                .toList();
        return new SourcesResponse(sources);
    }

    private static String buildDimensionInListFilter(String dimension, List<String> values) {
        return dimension + "=." + StreamEx.of(values)
                .map(s -> wrap(s, "'"))
                .joining(",", "(", ")");
    }

    /**
     * Получаем данные по сквозной аналитике из отчета Метрики "Источники, расходы и ROI".
     *
     * @param params Параметры запроса
     * @see #getTrafficSourceStatistics
     */
    @Override
    public StatisticsResponse getEndToEndStatistics(MetrikaByTimeStatisticsParams params) {
        String dateGroupBy = "day";
        String dimensions = "ym:ev:<attribution>ExpenseSource";

        boolean skipGoalData = Boolean.TRUE.equals(params.getSkipGoalData());
        checkArgument(!(skipGoalData && params.getGoalId() != null),
                "Suspect call: skipGoalData=true, goalId=%s", params.getGoalId());
        String goalVisitsMetricName =
                skipGoalData ? null :
                        params.getGoalId() != null
                                ? "ym:ev:goal<goal_id>visits"
                                : null;
        String conversionRateMetricName =
                skipGoalData || !params.getWithConversionRate() ? null :
                        params.getGoalId() != null
                                ? "ym:ev:goal<goal_id>conversionRate"
                                : "ym:ev:anyGoalConversionRate";
        String revenueMetricName =
                skipGoalData || !params.getWithRevenue() ? null :
                        params.getGoalId() != null
                                ? "ym:ev:goal<goal_id>ecommerce<currency>ConvertedRevenue"
                                : "ym:ev:ecommerce<currency>ConvertedRevenue";

        var metricsHandler = newMetricsHandler(
                "ym:ev:expenseClicks",
                conversionRateMetricName,
                goalVisitsMetricName,
                "ym:ev:expenses<currency>",
                revenueMetricName
        );

        return getByTimeStat(params, dateGroupBy, dimensions, metricsHandler);
    }

    /**
     * Получаем данные по источникам трафика из отчета Метрики "Источники, сводка".
     *
     * @param params Параметры запроса
     * @see #getEndToEndStatistics
     */
    @Override
    public StatisticsResponse getTrafficSourceStatistics(MetrikaByTimeStatisticsParams params) {
        String dateGroupBy = "day";
        String dimensions = "ym:s:<attribution>TrafficSource,ym:s:<attribution>SourceEngine";

        boolean skipGoalData = Boolean.TRUE.equals(params.getSkipGoalData());
        checkArgument(!(skipGoalData && params.getGoalId() != null),
                "Suspect call: skipGoalData=true, goalId=%s", params.getGoalId());
        String goalVisitsMetricName =
                skipGoalData ? null :
                        params.getGoalId() != null
                                ? "ym:s:goal<goal_id>visits"
                                : null;
        String conversionRateMetricName =
                skipGoalData || !params.getWithConversionRate() ? null :
                        params.getGoalId() != null
                                ? "ym:s:goal<goal_id>conversionRate"
                                : "ym:s:anyGoalConversionRate";
        String revenueMetricName =
                skipGoalData || !params.getWithRevenue() ? null :
                        params.getGoalId() != null
                                ? "ym:s:goal<goal_id>ecommerce<currency>ConvertedRevenue"
                                : "ym:s:ecommerce<currency>ConvertedRevenue";

        var metricsHandler = newMetricsHandler(
                "ym:s:visits",
                conversionRateMetricName,
                goalVisitsMetricName,
                null,
                revenueMetricName
        );

        return getByTimeStat(params, dateGroupBy, dimensions, metricsHandler);
    }

    private StatMetricsHandler<StatisticsResponseRow> newMetricsHandler(
            @Nullable String clicksMetricName,
            @Nullable String conversionRateMetricName,
            @Nullable String goalVisitsMetricName,
            @Nullable String expenseMetricName,
            @Nullable String revenueMetricName
    ) {
        StatMetricsHandler<StatisticsResponseRow> metricsHandler = StatMetricsHandler.withPeriodMapper(
                (row, period) -> row.setPeriod(period.toString())
        );
        if (clicksMetricName != null) {
            metricsHandler
                    .withMetricsMapper(clicksMetricName, (row, value) -> row.setClicks(value.longValue()));
        }
        if (conversionRateMetricName != null) {
            metricsHandler
                    .withMetricsMapper(conversionRateMetricName,
                            (row, value) -> row.setConversionRate(value.doubleValue()));
        }
        if (goalVisitsMetricName != null) {
            metricsHandler
                    .withMetricsMapper(goalVisitsMetricName, (row, value) -> row.setGoalVisits(value.longValue()));
        }
        if (expenseMetricName != null) {
            metricsHandler
                    .withMetricsMapper(expenseMetricName, StatisticsResponseRow::setExpenses);
        }
        if (revenueMetricName != null) {
            metricsHandler
                    .withMetricsMapper(revenueMetricName, StatisticsResponseRow::setRevenue);
        }
        return metricsHandler;
    }

    private StatisticsResponse getByTimeStat(
            MetrikaByTimeStatisticsParams params,
            String dateGroupBy,
            String dimensions,
            StatMetricsHandler<StatisticsResponseRow> metricsHandler
    ) {
        var attribution = params.getAttribution();
        var attributionStr = attribution == null ? null : attribution.name();
        var currencyCode = params.getCurrencyCode();
        var currencyCodeStr = currencyCode == null ? null : currencyCode.toString();

        var rowIds = params.getRowIds();
        var rowIdsStr = rowIds != null ? convertRowIdsToString(rowIds) : null;

        Result<GetStatByDateResult> rawResult =
                metrikaApi.getByTimeStat(
                        params.getCounterId(),
                        dateGroupBy,
                        attributionStr,
                        currencyCodeStr,
                        dimensions,
                        metricsHandler.metricsQueryParams(),
                        params.getGoalId(),
                        params.getDateFrom().toString(),
                        params.getDateTo().toString(),
                        rowIdsStr,
                        params.getChiefLogin()
                ).execute();

        checkResultForErrors(rawResult, MetrikaClientException::new);

        GetStatByDateResult result = rawResult.getSuccess();

        var statisticsResponseRows =
                metricsHandler.resultToStatRows(StatisticsResponseRow::new, result);
        if (params.getGoalId() == null) {
            calculateGoalVisitsByConversionRates(statisticsResponseRows);
        }
        var rowsWithTotals = appendTotals(statisticsResponseRows, params.getWithConversionRate());

        return new StatisticsResponse(result.getQuery().getCurrency(), rowsWithTotals);
    }

    private String convertRowIdsToString(List<List<String>> rowIds) {
        return StreamEx.of(rowIds)
                .map(this::convertRowIdsToStringSingle)
                .joining(",", "[", "]");
    }

    private String convertRowIdsToStringSingle(List<String> rowId) {
        return StreamEx.of(rowId)
                .map(s -> wrap(s, '"'))
                .joining(",", "[", "]");
    }

    private void calculateGoalVisitsByConversionRates(List<StatisticsResponseRow> rows) {
        for (var row : rows) {
            var clicks = row.getClicks();
            var conversionRate = row.getConversionRate();
            var goalVisits = (clicks != null && conversionRate != null) ? conversionRate * clicks / 100 : null;
            row.setGoalVisits(goalVisits != null ? Math.round(goalVisits) : null);
        }
    }

    private List<StatisticsResponseRow> appendTotals(
            List<StatisticsResponseRow> statisticsResponseRows,
            boolean aggregateConversionRate
    ) {
        // Для красоты сохраняем примерный порядок и total'ы складываем рядом с группой
        Map<List<Dimension>, List<StatisticsResponseRow>> groupedByDimensions =
                StreamEx.of(statisticsResponseRows)
                        .groupingBy(StatisticsResponseRow::getDimensions, LinkedHashMap::new, Collectors.toList());
        int totalRowsSize = groupedByDimensions.size() + statisticsResponseRows.size();
        List<StatisticsResponseRow> withTotals = new ArrayList<>(totalRowsSize);
        for (var entry : groupedByDimensions.entrySet()) {
            StatisticsResponseRow total = new StatisticsResponseRow(entry.getKey());
            List<StatisticsResponseRow> group = entry.getValue();
            // Если значение не получили, сохраняем null в total с помощью nullKeeping*Reducer()
            var clicks = group.stream()
                    .map(StatisticsResponseRow::getClicks)
                    .reduce(null, nullKeepingLongReducer());
            total.setClicks(clicks);
            var goalVisits = group.stream()
                    .map(StatisticsResponseRow::getGoalVisits)
                    .reduce(null, nullKeepingLongReducer());
            total.setGoalVisits(goalVisits);
            var conversionRate = (aggregateConversionRate && clicks != null && goalVisits != null)
                    ? goalVisits * 100.0 / clicks
                    : null;
            total.setConversionRate(conversionRate);
            total.setExpenses(
                    group.stream()
                            .map(StatisticsResponseRow::getExpenses)
                            .reduce(null, nullKeepingBigDecimalReducer()));
            total.setRevenue(
                    group.stream()
                            .map(StatisticsResponseRow::getRevenue)
                            .reduce(null, nullKeepingBigDecimalReducer()));
            withTotals.addAll(group);
            withTotals.add(total);
        }
        return withTotals;
    }

    private BinaryOperator<BigDecimal> nullKeepingBigDecimalReducer() {
        return (acc, value) -> {
            if (acc == null) {
                return value;
            }
            if (value != null) {
                return acc.add(value);
            }
            return acc;
        };
    }

    private BinaryOperator<Long> nullKeepingLongReducer() {
        return (acc, value) -> {
            if (acc == null) {
                return value;
            }
            if (value != null) {
                return acc + value;
            }
            return acc;
        };
    }

    @Override
    public Map<Long, GoalConversionInfo> getGoalsConversionInfoByCounterIds(Collection<Integer> counterIds, int days) {
        LocalDate now = getLocalDate();
        return getGoalsConversionInfoByCounterIds(counterIds, now.minus(Period.ofDays(days)), now);
    }

    @Override
    public Map<Long, GoalConversionInfo> getGoalsConversionInfoByCounterIds(
            Collection<Integer> counterIds, LocalDate dateFrom, LocalDate dateTo) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        List<Integer> ids = List.copyOf(counterIds);
        String dateFromStr = METRIKA_DATE_FORMAT.format(dateFrom);
        String dateToStr = METRIKA_DATE_FORMAT.format(dateTo);
        Collection<Result<GoalConversionsCountResponse>> results = intApi.getGoalCounts(ids, dateFromStr, dateToStr)
                .execute(100, parallelRequestsProperty.getOrDefault(1)).values();
        results.forEach(result ->
                checkResultForErrors(result, MetrikaClientException::new, ids, dateFromStr, dateToStr));

        return StreamEx.of(results)
                .flatMap(result -> result.getSuccess().getResponse().stream())
                .flatMap(item -> item.getCounts().stream())
                .toMap(GoalConversionInfo::getGoalId, Function.identity());
    }

    @Override
    public Map<Long, Long> getGoalsStatistics(List<Integer> counterIds, LocalDate dateFrom, LocalDate dateTo) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        List<Integer> ids = List.copyOf(counterIds);
        String dateFromStr = METRIKA_DATE_FORMAT.format(dateFrom);
        String dateToStr = METRIKA_DATE_FORMAT.format(dateTo);
        Collection<Result<GetStatResult>> results = metrikaApi
                .getGoalsStatistics(ids, GOALS_STATISTIC_LIMIT, dateFromStr, dateToStr)
                .execute(MAX_COUNTERS_SIZE, parallelRequestsProperty.getOrDefault(1)).values();

        results.forEach(result -> checkResultForErrors(
                result, MetrikaClientException::new, ids, dateFromStr, dateToStr));

        List<StatData> goalsStatData = flatMap(results, result -> result.getSuccess().getData());
        return listToMap(goalsStatData, StatData::getFirstDimensionIdAsLong, StatData::getMetricAsLong);
    }

    @Override
    public Map<Integer, List<CounterGoal>> getMassCountersGoalsFromMetrika(Set<Integer> counterIds) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        Collection<Result<GetCountersGoalsResponse>> resultByCounterIdSubList = intApi.getCountersGoals(counterIds)
                .execute(MAX_COUNTERS_SIZE, parallelRequestsProperty.getOrDefault(1)).values();

        if (resultByCounterIdSubList.stream().anyMatch(this::notFoundResponse)) {
            return emptyMap();
        }
        resultByCounterIdSubList.forEach(result -> checkResultForErrors(result, MetrikaClientException::new));

        List<CounterGoalExtended> rawGoals = StreamEx.of(resultByCounterIdSubList)
                .map(Result::getSuccess)
                .flatMap(counterGoal -> counterGoal.getGoals().stream())
                .removeBy(goal -> goal.getGoal().getType(), CounterGoal.Type.UNKNOWN)
                .removeBy(goal -> goal.getGoal().getGoalSource(), CounterGoal.Source.UNKNOWN)
                .toList();

        Map<Integer, List<CounterGoal>> stepGoalsByCounterId = StreamEx.of(rawGoals)
                .mapToEntry(Function.identity(), counterGoal -> counterGoal.getGoal().getSteps())
                .removeValues(CollectionUtils::isEmpty)
                .flatMapValues(Collection::stream)
                .mapToValue((counterGoal, stepGoal) -> new CounterGoal()
                        .withId((int) stepGoal.getId())
                        .withParentId(counterGoal.getGoal().getId())
                        .withType(stepGoal.getType())
                        .withStatus(stepGoal.getStatus())
                        .withSource(stepGoal.getGoalSource())
                        .withDefaultPrice(stepGoal.getDefaultPrice())
                        .withName(stepGoal.getName()))
                .mapKeys(CounterGoalExtended::getCounterId)
                .grouping();

        return StreamEx.of(rawGoals)
                .mapToEntry(CounterGoalExtended::getCounterId, CounterGoalExtended::getGoal)
                .mapValues(goal -> new CounterGoal()
                        .withId((int) goal.getId())
                        .withType(goal.getType())
                        .withStatus(goal.getStatus())
                        .withSource(goal.getGoalSource())
                        .withDefaultPrice(goal.getDefaultPrice())
                        .withName(goal.getName()))
                .append(EntryStream.of(stepGoalsByCounterId).flatMapValues(Collection::stream))
                .grouping();
    }

    private boolean notFoundResponse(Result<?> result) {
        if (!isEmpty(result.getErrors()) && result.getErrors().size() == 1) {
            Throwable throwable = result.getErrors().get(0);
            if (throwable instanceof ErrorResponseWrapperException) {
                var ex = (ErrorResponseWrapperException) throwable;
                return ex.getResponse() != null && ex.getResponse().getStatusCode() == 404;
            }
        }
        return false;
    }

    @Override
    public UpdateCounterGrantsResponse updateCounterGrants(long counterId, Set<String> logins) {
        UpdateCounterGrantsMetrikaRequest updateCounterGrantsMetrikaRequest = new UpdateCounterGrantsMetrikaRequest()
                .withCounter(new UpdateCounterGrantsMetrikaRequestCounter()
                        .withGrants(mapList(logins,
                                login -> new UpdateCounterGrantsMetrikaRequestGrant()
                                        .withUserLogin(login)
                                        .withPerm("view"))
                        )
                );
        Result<Object> result = intApi
                .updateCounterGrants(
                        counterId,
                        "1",
                        updateCounterGrantsMetrikaRequest)
                .execute();

        logger.info("update grants for counterId={}, logins={}", counterId, logins);
        if (result.getErrors() != null) {
            logger.warn("errors in response={} for counterId={}, logins={}", result.getErrors(), counterId, logins);
            return new UpdateCounterGrantsResponse()
                    .withSuccessful(false)
                    .withMetrikaErrorResponse(getMetrikaErrorResponse(result.getErrors()));
        }
        return new UpdateCounterGrantsResponse()
                .withSuccessful(true);
    }

    @Nullable
    private static MetrikaErrorResponse getMetrikaErrorResponse(@Nullable List<Throwable> throwableList) {
        if (throwableList == null) {
            return null;
        }
        Throwable errorResponseWrapperException = throwableList.stream()
                .filter(throwable -> throwable instanceof ErrorResponseWrapperException)
                .findAny()
                .orElse(null);
        return errorResponseWrapperException != null ?
                getMetrikaErrorResponse((ErrorResponseWrapperException) errorResponseWrapperException) : null;
    }

    @Nullable
    private static MetrikaErrorResponse getMetrikaErrorResponse(ErrorResponseWrapperException throwable) {
        String responseBody = ifNotNull(throwable.getResponse(), Response::getResponseBody);
        if (responseBody != null) {
            return JsonUtils.fromJson(responseBody, MetrikaErrorResponse.class);
        }
        return null;
    }

    @Override
    public RequestGrantsResponse requestCountersGrants(RequestGrantsRequest request) {
        var result = intApi.requestGrants(request).execute();

        checkResultForErrors(result, MetrikaClientException::new, request);

        return result.getSuccess();
    }

    @Override
    public GrantAccessRequestStatusesResponse getGrantAccessRequestStatuses(Long uid,
                                                                            GrantAccessRequestStatusesRequest request) {
        var result = intApi.getGrantAccessRequestStatuses(uid, request).execute();

        checkResultForErrors(result, MetrikaClientException::new, uid, request);

        return result.getSuccess();
    }

    @Override
    public GetExistentCountersResponse getExistentCounters(GetExistentCountersRequest request) {
        var result = intApi.getExistentCounters(request).execute();
        checkResultForErrors(result, MetrikaClientException::new, request);

        return result.getSuccess();
    }

    @Override
    public TurnOnCallTrackingResponse turnOnCallTracking(Long counterId) {
        Result<TurnOnCallTrackingResponse> result = intApi.turnOnCallTracking(counterId).execute();
        checkResultForErrors(result, MetrikaClientException::new, counterId);

        return result.getSuccess();
    }

    @Override
    public List<Counter> getEditableCounters(@Nullable String tvmUserTicket) {
        var authHeaders = makeAuthHeaders(tvmUserTicket);

        // own - свои счётчики, edit - гостевые счётчики с правами на редактирование
        String permission = "own,edit";
        var editableCounters = metrikaApi
                .getCounters(permission, authHeaders)
                .execute();
        checkResultForErrors(editableCounters, MetrikaClientException::new, permission);

        return editableCounters.getSuccess().getCounters();
    }

    @Override
    public Counter getCounter(Long counterId) {
        checkNotNull(counterId);
        Result<GetCounterResponseEnvelope> result = metrikaApi.getCounter(counterId).execute();
        checkResultForErrors(result, MetrikaClientException::new, counterId);
        return result.getSuccess().getCounter();
    }

    @Override
    public List<Segment> getSegments(int counterId, @Nullable String tvmUserTicket) {
        var authHeaders = makeAuthHeaders(tvmUserTicket);

        var result = metrikaApi.getSegments(counterId, authHeaders).execute();
        checkResultForErrors(result, MetrikaClientException::new, counterId);

        return result.getSuccess().getSegments();
    }

    @Override
    public Segment createSegment(int counterId, String name, String expression, @Nullable String tvmUserTicket) {
        var authHeaders = makeAuthHeaders(tvmUserTicket);

        var segment = new CreateSegmentRequestEnvelope.SegmentData()
                .withName(name)
                .withExpression(expression);
        var body = new CreateSegmentRequestEnvelope().withSegment(segment);

        var result = metrikaApi.createSegment(counterId, authHeaders, body).execute();
        checkResultForErrors(result, MetrikaClientException::new, counterId, body);

        return result.getSuccess().getSegment();
    }

    @Override
    public CreateCounterGoal createGoal(int counterId, CreateCounterGoal goalToCreate, @Nullable String tvmUserTicket) {
        var authHeaders = makeAuthHeaders(tvmUserTicket);

        var body = new CreateGoalRequestEnvelope()
                .withGoal(goalToCreate);

        var result = metrikaApi.createGoals(counterId, authHeaders, body).execute();
        checkResultForErrors(result, MetrikaClientException::new, counterId, body);

        return result.getSuccess().getGoal();
    }

    private static Map<String, String> makeAuthHeaders(@Nullable String tvmUserTicket) {
        var result = new HashMap<String, String>();
        if (tvmUserTicket != null) {
            result.put(TvmHeaders.USER_TICKET, tvmUserTicket);
        }
        return result;
    }
}
