package ru.yandex.direct.metrika.client.asynchttp

import ru.yandex.direct.metrika.client.internal.Dimension
import ru.yandex.direct.metrika.client.internal.GetStatByDateResult
import java.math.BigDecimal
import java.time.LocalDate
import java.util.function.BiConsumer
import java.util.function.Function

/**
 * Класс для удобной работы с [MetrikaApi.getByTimeStat].
 *
 * Умеет сохранять запрашиваемые метрики и парсить результат.
 */
class StatMetricsHandler<T> private constructor(
    val periodMapper: BiConsumer<T, LocalDate>
) {
    private val metricsNames: MutableList<String> = mutableListOf()
    private val metricsMappers: MutableList<BiConsumer<T, BigDecimal>> = mutableListOf()

    companion object {
        @JvmStatic
        fun <T> withPeriodMapper(mapper: BiConsumer<T, LocalDate>): StatMetricsHandler<T> {
            return StatMetricsHandler(mapper)
        }
    }

    fun withMetricsMapper(
        name: String,
        mapper: BiConsumer<T, BigDecimal>
    ): StatMetricsHandler<T> {
        metricsNames.add(name)
        metricsMappers.add(mapper)
        return this
    }

    fun metricsQueryParams() = metricsNames.joinToString(",")

    fun resultToStatRows(
        rowCreator: Function<List<Dimension>, T>,
        statResult: GetStatByDateResult
    ): List<T> {
        val localPeriodMapper = periodMapper

        val dataByDimension = statResult.data
        val timeIntervals = statResult.timeIntervals

        val result = mutableListOf<T>()
        for (dimensionData in dataByDimension) {
            val metrics = dimensionData.metrics!!
            for ((intervalIdx, interval) in timeIntervals.withIndex()) {
                val row = rowCreator.apply(dimensionData.dimensions!!)
                val period = interval[0]
                localPeriodMapper.accept(row, period)
                for ((i, mapper) in metricsMappers.withIndex()) {
                    mapper.accept(row, metrics[i][intervalIdx] ?: BigDecimal.ZERO)
                }
                result.add(row)
            }
        }

        return result
    }
}
