package ru.yandex.direct.libs.mirrortools.utils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.utils.UrlUtils;

/**
 * Class to operate with hostings, domains, urls etc as strings
 */

@ParametersAreNonnullByDefault
public class HostingsHandler {
    private static final String WWW_PREFIX = "www.";
    private static final Pattern PROTOCOL_PATTERN = Pattern.compile("^https?://");
    private static final CharMatcher SOME_DOMAIN_MATCHER =
            CharMatcher.inRange('0', '9').or(CharMatcher.inRange('a', 'z')).or(CharMatcher.is('-'));

    private static final CharMatcher DOMAIN_TAIL_MATCHER = CharMatcher.anyOf("/?");
    private static final Pattern PORT_PATTERN = Pattern.compile("(:[0-9]+)$");

    private final Set<String> hostings;
    private final Set<String> publicSecondLevelDomains;

    /**
     * @param hostings                 - List of domains which can provide third(and more) level domains based on them.
     *                                 Usually it's free hostings, blogs, portals, agency and companies that place ad
     *                                 in web.
     * @param publicSecondLevelDomains - List of known public second level domains
     * @see
     * <a href="https://svn.yandex-team.ru/websvn/wsvn/direct-utils/yandex-lib/mirrortools-hostings/lib/Yandex/MirrorsTools/Hostings.pm">Yandex::MirrorTools::Hostings
     * </a> — "@hostings"
     * @see
     * <a href="https://svn.yandex-team.ru/websvn/wsvn/direct-utils/yandex-lib/mirrortools-hostings/lib/Yandex/MirrorsTools/Hostings.pm">Yandex::MirrorTools::Hostings
     * </a> — "@public2ld"
     */
    public HostingsHandler(List<String> hostings, List<String> publicSecondLevelDomains) {
        this.hostings = ImmutableSet.copyOf(hostings);
        this.publicSecondLevelDomains = ImmutableSet.copyOf(publicSecondLevelDomains);
    }

    public static String stripDomainTail(String domain) {
        int pos = DOMAIN_TAIL_MATCHER.indexIn(domain);
        if (pos >= 0) {
            return domain.substring(0, pos);
        } else {
            return domain;
        }
    }

    /**
     * Удалить порт с конца строки
     */
    public String stripPort(String domain) {
        return PORT_PATTERN.matcher(domain).replaceFirst("");
    }

    /**
     * Удалить протокол http/https
     */
    public static String stripProtocol(String domain) {
        return PROTOCOL_PATTERN.matcher(domain).replaceFirst("");
    }

    /**
     * возвращает "владельца" сайта с учетом популярных хостингов
     * <p>
     * если сайт заканчивается на поплуярный хостинг (например, tilda.ws), то возвращается домен на один уровнь
     * ниже популярного хостинга (mysite.tilda.ws)
     * <p>
     * если домен второго уровня сайта публичный (com.ru), возвращается домен на один уровнь ниже
     * публичного домена (mysite.com.ru)
     * <p>
     * инча возвращается 2 верних уровня домена (mysite.ru)
     * аналог perl https://a.yandex-team.ru/arc/trunk/arcadia/direct/infra/direct-utils/yandex-lib/mirrortools
     * -hostings/lib/Yandex/MirrorsTools/Hostings.pm?rev=7490054#L315
     */
    public String getHosting(String site) {
        var domain = UrlUtils.trimPort(stripDomainTail(stripProtocol(site.toLowerCase())));

        if (!domain.contains(".")) {
            return domain;
        }

        /* заканчивается ли домен на известный хостинг */
        var popularHosting = takePopularHosting(domain, hostings);
        if (Objects.nonNull(popularHosting)) {
            return getNextLevelDomain(domain, popularHosting);
        }

        /* Является ли домен второго уровня  публичным */
        var publicSecondLevelDomain = getPublicSecondLevelDomain(domain);
        if (Objects.nonNull(publicSecondLevelDomain)) {
            return getNextLevelDomain(domain, publicSecondLevelDomain);
        }

        var topLevelDomain = StringUtils.substringAfterLast(domain, ".");
        return getNextLevelDomain(domain, topLevelDomain);
    }

    /**
     * Возвращает домен на один уровень больше указанного
     * Если fullLevelsDomain заканчивается не на указанный domain или не домен уровнем выше не удовлетворяет
     * {@link #SOME_DOMAIN_MATCHER}, то преобразование не применяется
     *
     * @param fullLevelsDomain домен с полным набором поддоменов
     * @param domain           домен, к котрому добавляется уровнь
     */
    private String getNextLevelDomain(String fullLevelsDomain, String domain) {
        var fullLevelDomainWithoutSuffix = StringUtils.removeEnd(fullLevelsDomain, domain);
        if (!fullLevelDomainWithoutSuffix.endsWith(".")) {
            return fullLevelsDomain;
        }
        fullLevelDomainWithoutSuffix = StringUtils.removeEnd(fullLevelDomainWithoutSuffix, ".");
        var subdomains = fullLevelDomainWithoutSuffix.split("\\.");
        var highLevelSubdomain = subdomains[subdomains.length - 1];
        return highLevelSubdomain + "." + domain;
    }

    /**
     * Если домен находится на популярном хостинге, возвращает его
     * Иначе - null
     */
    private String takePopularHosting(String input, Collection<String> suffixes) {
        return suffixes.stream().filter(input::endsWith).findFirst().orElse(null);
    }

    /**
     * Если домен второго уровня публичный, возвращает 2 верхних домена
     * Иначе - null
     */
    private String getPublicSecondLevelDomain(String domain) {
        var candidateLastIndex = domain.lastIndexOf(".");
        if (candidateLastIndex < 0) {
            return null;
        }
        var topLevelDomain = StringUtils.substringAfterLast(domain, ".");
        var domainWithoutTopLevel = StringUtils.substringBeforeLast(domain, ".");
        var subdomains = domainWithoutTopLevel.split("\\.");
        if (subdomains.length == 1) {
            return null;
        }
        var secondLevelDomain = subdomains[subdomains.length - 1];

        if (publicSecondLevelDomains.contains(secondLevelDomain)) {
            return secondLevelDomain + "." + topLevelDomain;
        } else {
            return null;
        }
    }

    /**
     * Strips "www" from domain if it's unnecessary
     *
     * @param domain - input domain string. Can go with protocol/path/query
     * @return — domain without "www" if it's unnecessary
     * @see
     * <a href="https://svn.yandex-team.ru/websvn/wsvn/direct-utils/yandex-lib/mirrortools-hostings/lib/Yandex/MirrorsTools/Hostings.pm">Yandex::MirrorTools::Hostings
     * </a> — "strip_www"
     * <p>
     */
    public String stripWww(String domain) {
        domain = stripProtocol(domain);
        domain = domain.toLowerCase();
        domain = stripDomainTail(domain);

        if (!domain.startsWith(WWW_PREFIX)) {
            return domain;
        }

        String candidate = domain.substring(WWW_PREFIX.length());
        if (hostings.contains(candidate)) {
            return domain;
        }

        int lastDotPos = candidate.lastIndexOf('.');
        if (lastDotPos < 0) {

            if (SOME_DOMAIN_MATCHER.matchesAllOf(candidate)) {
                return domain;
            } else {
                return candidate;
            }
        }

        if (publicSecondLevelDomains.contains(candidate.substring(0, lastDotPos))) {
            if (SOME_DOMAIN_MATCHER.matchesAllOf(candidate.substring(lastDotPos + 1))) {
                return domain;
            } else {
                return candidate;
            }
        }

        return candidate;
    }

    /**
     * Является ли домен первым под общественным доменом второго уровня или под известным хостингом.
     * Не расчитан на работу с меткой абсолютного домена (точка в конце доменной записи)
     *
     * @param domain — домен. Применяются те же преобразования, что и в stripWww
     * @return является или не является
     */
    public boolean isFirstSubdomainOfPublicDomainOrHosting(String domain) {
        domain = stripWww(domain);
        int firstDotPosition = domain.indexOf('.');
        if (domain.length() - firstDotPosition < 2) { // после первой точки нет ничего
            return false;
        }
        String higherDomains = domain.substring(firstDotPosition + 1);
        firstDotPosition = higherDomains.indexOf('.');
        String nextHigherDomain = firstDotPosition > 0 ? higherDomains.substring(0, firstDotPosition) : higherDomains;
        return (getDomainLevel(higherDomains) == 2 && publicSecondLevelDomains.contains(nextHigherDomain))
                || hostings.contains(higherDomains);
    }

    /**
     * Какой уровень у домена
     *
     * @param domain — домен
     * @return уровень домена, где TLD имеют уровень 1
     */
    public static int getDomainLevel(String domain) {
        return domain.split("\\.").length;
    }

    public Set<String> getHostings() {
        return hostings;
    }

    public Set<String> getPublicSecondLevelDomains() {
        return publicSecondLevelDomains;
    }
}



