package ru.yandex.direct.model.generator;

import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import com.google.common.base.Splitter;
import com.squareup.javapoet.JavaFile;
import one.util.streamex.StreamEx;
import org.reflections.Reflections;
import org.reflections.scanners.ResourcesScanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.model.generator.old.conf.ModelConfFactory;
import ru.yandex.direct.model.generator.old.conf.UpperLevelModelConf;
import ru.yandex.direct.model.generator.old.javafile.JavaFileFactory;
import ru.yandex.direct.model.generator.old.javafile.Util;
import ru.yandex.direct.model.generator.old.registry.ModelConfRegistry;
import ru.yandex.direct.model.generator.old.spec.factory.JavaFileSpecFactory;
import ru.yandex.direct.model.generator.rewrite.ModelRegistry;
import ru.yandex.direct.model.generator.rewrite.PropertyResolver;
import ru.yandex.direct.model.generator.rewrite.builder.Builder;
import ru.yandex.direct.model.generator.rewrite.conf.ConfParser;
import ru.yandex.direct.model.generator.rewrite.conf.UpperLevelConf;
import ru.yandex.direct.model.generator.rewrite.spec.SpecFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * CLI-программа, генерируючая .java классы по .conf файлам.
 */
public class Tool {
    private static final String CONF_SUFFIX = ".conf";

    private final ToolParams params;
    private final PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher("glob:*" + CONF_SUFFIX);
    private final Set<String> written = new HashSet<>();
    private final Path outputPath;

    private final ModelConfFactory modelConfFactory = new ModelConfFactory();
    private final JavaFileFactory javaFileFactory = new JavaFileFactory();

    private final ConfParser configFactory = new ConfParser();
    private final Builder builder = new Builder();

    private Tool(String[] args) {
        params = new ToolParams();
        ParserWithHelp.parse(Tool.class.getCanonicalName(), args, params);
        outputPath = Paths.get(params.outputPath);
    }

    public static void main(String[] args) throws Exception {
        new Tool(args).run();
    }

    private static URL toURL(URI uri) {
        try {
            return uri.toURL();
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    private void run() throws IOException {
        final List<File> sourceFiles = new ArrayList<>();
        List<URL> classPathConfigs = new ArrayList<>();

        if (params.prefixToSearchConfInClasspath != null) {
            var conf = new ConfigurationBuilder()
                    .setUrls(ClasspathHelper.forPackage(params.prefixToSearchConfInClasspath))
                    .setScanners(new ResourcesScanner());
            Reflections reflections = new Reflections(conf);

            Set<String> resources = reflections.getResources(name -> name.endsWith(CONF_SUFFIX));
            for (String resource : resources) {
                ClasspathHelper.contextClassLoader().getResources(resource)
                        .asIterator()
                        .forEachRemaining(classPathConfigs::add);
            }
        }

        var classPathModelConfStream = classPathConfigs.stream()
                .sorted(Comparator.comparing(URL::getFile));

        for (String confDir : params.confDirs) {
            try (Stream<Path> walk = Files.walk(Paths.get(confDir))) {
                walk.filter(p -> pathMatcher.matches(p.getFileName()))
                        .map(Path::toFile)
                        .forEach(sourceFiles::add);
            }
        }

        Collection<URL> fileUrls = StreamEx.of(sourceFiles)
                .sorted(Comparator.comparing(File::getAbsolutePath))
                .map(File::toURI)
                .map(Tool::toURL)
                .prepend(classPathModelConfStream)
                .toList();

        if (!Boolean.TRUE.equals(params.rewrite)) {
            Collection<UpperLevelModelConf> modelConfigs = StreamEx.of(fileUrls)
                    .map(modelConfFactory::createModelConf)
                    .toList();

            ModelConfRegistry registry = new ModelConfRegistry(modelConfigs);
            JavaFileSpecFactory specFactory = new JavaFileSpecFactory(registry);

            specFactory.convertAllConfigsToSpecs().stream()
                    .map(javaFileFactory::buildJavaFile)
                    .forEach(this::write);
        } else {
            Collection<UpperLevelConf> modelConfigs = StreamEx.of(fileUrls)
                    .map(configFactory::parseConfFile)
                    .toList();

            ModelRegistry registry = new ModelRegistry(modelConfigs);
            PropertyResolver propertyResolver = new PropertyResolver(registry);
            SpecFactory specFactory = new SpecFactory(registry, propertyResolver);

            specFactory.convertAllConfigsToSpecs(registry.allConfs()).stream()
                    .map(builder::buildJavaFile)
                    .forEach(this::write);
        }
    }

    private void write(JavaFile javaFile) {
        String className = javaFile.packageName + "." + javaFile.typeSpec.name;
        checkState(written.add(className), "Attempt of double write of %s", className);
        String code = Util.rearrangeImports(javaFile.toString());
        code = Util.trimTrailingSpaces(code);
        try (OutputStreamWriter writer = createWriter(javaFile)) {
            writer.write(code);
        } catch (IOException e) {
            throw new IllegalStateException("Can't write to file: " + outputPath, e);
        }
    }

    // копипаста из writeTo(Path directory)
    private OutputStreamWriter createWriter(JavaFile javaFile) throws IOException {
        Path dir = outputPath;
        if (!javaFile.packageName.isEmpty()) {
            for (String packageComponent : Splitter.on('.').split(javaFile.packageName)) {
                dir = dir.resolve(packageComponent);
            }
            Files.createDirectories(dir);
        }

        Path outputPath = dir.resolve(javaFile.typeSpec.name + ".java");

        return new OutputStreamWriter(Files.newOutputStream(outputPath), UTF_8);
    }
}
