package ru.yandex.direct.model.generator.old.conf;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.ClassName;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.util.AttrNameUtil;
import ru.yandex.direct.model.generator.old.util.FullClassNameProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public abstract class AbstractModelConf implements FullClassNameProvider {

    static class SourceProperty {
        static final String ATTRS = "attrs";
        static final String COMMENT = "comment";
        static final String ENUMS = "enums";
        static final String ANNOTATIONS = "annotated_by";
        static final String EXTENDS = "extends";
        static final String GENERATE_PROPERTIES = "generateProperties";
        static final String GENERATE_COPY_METHOD = "generateCopyMethod";
        static final String JSON_SUBTYPES = "jsonSubtypes";
        static final String JSON_SUBTYPES_WITH_NAME_VALUE = "jsonSubtypesWithNameValue";
        static final String IMPLEMENTS = "implements";
        static final String INTERFACES = "interfaces";
        static final String MODIFIERS = "modifiers";
        static final String NAME = "name";
        static final String PACKAGE = "package";
        static final String PACKAGE_NAME = "packageName";

        private SourceProperty() {
        }
    }

    private final String source;
    private final String packageName;
    private final String name;
    private final String comment;
    private final List<AttrConf> attrs;
    private final List<AnnotationConf> annotations;
    private final boolean generateProperties;
    private final boolean jsonSubtypes;
    private final boolean jsonSubtypesWithNameValue;

    private ClassName className;

    @SuppressWarnings("checkstyle:parameternumber")
    AbstractModelConf(String source, String packageName, String name, String comment,
                      List<AttrConf> attrs, List<AnnotationConf> annotations,
                      boolean generateProperties, boolean jsonSubtypes, boolean jsonSubtypesWithNameValue
    ) {
        if (generateProperties) {
            List<AttrConf> primitiveAttrs = StreamEx.of(attrs).filter(a -> a.getType().isPrimitive()).toList();
            if (!primitiveAttrs.isEmpty()) {
                throw new IllegalArgumentException("generateProperties must be disabled, while we have primitives: "
                        + StreamEx.of(primitiveAttrs).map(AttrConf::getName).joining(", "));
            }
        }
        this.source = source;
        this.packageName = checkNotNull(packageName);
        this.name = checkNotNull(name);
        this.comment = checkNotNull(comment);
        this.attrs = unmodifiableList(attrs);
        this.annotations = unmodifiableList(annotations);
        this.generateProperties = generateProperties;
        this.jsonSubtypes = jsonSubtypes;
        this.jsonSubtypesWithNameValue = jsonSubtypesWithNameValue;

        this.className = ClassName.get(this.packageName, this.name);
    }

    @Nullable
    public String getSourceFile() {
        return source;
    }

    public ClassName getClassName() {
        return className;
    }

    public String getPackageName() {
        return packageName;
    }

    public String getName() {
        return name;
    }

    public String getComment() {
        return comment;
    }

    public List<AttrConf> getAttrs() {
        return attrs;
    }

    public List<String> getAttrNames() {
        return mapList(attrs, AttrConf::getName);
    }

    public List<AnnotationConf> getAnnotations() {
        return annotations;
    }

    /**
     * @return {@code true}, если в классе/интерфейсе должны быть доступны статические
     * {@link ru.yandex.direct.model.ModelProperty}.
     */
    public boolean isGenerateProperties() {
        return generateProperties;
    }

    public boolean isJsonSubtypes() {
        return jsonSubtypes;
    }

    public boolean isJsonSubtypesWithNameValue() {
        return jsonSubtypesWithNameValue;
    }

    @Override
    public String getFullName() {
        return packageName + "." + name;
    }

    public List<String> getAttributesFullPaths() {
        return attrs.stream().map(a -> AttrNameUtil.getAttrFullName(a, this)).collect(toList());
    }

    public List<String> getNestedModelsAttributesFullPaths() {
        return emptyList();
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("AbstractModelConf{");
        sb.append("source='").append(source).append('\'');
        sb.append(", packageName='").append(packageName).append('\'');
        sb.append(", name='").append(name).append('\'');
        sb.append(", comment='").append(comment).append('\'');
        sb.append(", attrs=").append(attrs);
        sb.append(", annotations=").append(annotations);
        sb.append(", generateProperties=").append(generateProperties);
        sb.append(", jsonSubtypes=").append(jsonSubtypes);
        sb.append(", jsonSubtypesWithNameValue=").append(jsonSubtypesWithNameValue);
        sb.append(", className=").append(className);
        sb.append(", sourceFile='").append(getSourceFile()).append('\'');
        sb.append(", fullName='").append(getFullName()).append('\'');
        sb.append(", attributesFullPaths=").append(getAttributesFullPaths());
        sb.append('}');
        return sb.toString();
    }

    @Override
    @SuppressWarnings("EqualsGetClass")
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AbstractModelConf modelConf = (AbstractModelConf) o;
        return generateProperties == modelConf.generateProperties &&
                jsonSubtypes == modelConf.jsonSubtypes &&
                jsonSubtypesWithNameValue == modelConf.jsonSubtypesWithNameValue &&
                Objects.equals(source, modelConf.source) &&
                Objects.equals(packageName, modelConf.packageName) &&
                Objects.equals(name, modelConf.name) &&
                Objects.equals(comment, modelConf.comment) &&
                Objects.equals(attrs, modelConf.attrs) &&
                Objects.equals(className, modelConf.className);
    }

    @Override
    public int hashCode() {
        return Objects.hash(source, packageName, name, comment, attrs, generateProperties, jsonSubtypes, jsonSubtypesWithNameValue, className);
    }

    abstract static class AbstractBuilder {
        private String sourceFileName;
        private String pack;
        private String name;
        private String comment = "";
        private List<AttrConf> attrs;
        private List<AnnotationConf> annotations;
        private boolean generateProperties = true;
        private boolean jsonSubtypes = false;
        private boolean jsonSubtypesWithNameValue = false;

        AbstractBuilder(String pack, String name) {
            this.pack = pack;
            this.name = name;
        }

        String getSource() {
            return sourceFileName;
        }

        String getPack() {
            return pack;
        }

        String getName() {
            return name;
        }

        String getComment() {
            return comment;
        }

        List<AttrConf> getAttrs() {
            return attrs;
        }

        List<AnnotationConf> getAnnotations() {
            return annotations;
        }

        boolean isGenerateProperties() {
            return generateProperties;
        }

        boolean isJsonSubtypes() {
            return jsonSubtypes;
        }

        boolean isJsonSubtypesWithNameValue() {
            return jsonSubtypesWithNameValue;
        }

        AbstractBuilder withSource(String sourceFileName) {
            this.sourceFileName = sourceFileName;
            return this;
        }

        AbstractBuilder withComment(String comment) {
            this.comment = comment;
            return this;
        }

        AbstractBuilder withAttrs(List<AttrConf> attrs) {
            this.attrs = attrs;
            return this;
        }

        AbstractBuilder withGenerateProperties(boolean generateProperties) {
            this.generateProperties = generateProperties;
            return this;
        }

        AbstractBuilder withJsonSubtypes(boolean jsonSubtypes) {
            this.jsonSubtypes = jsonSubtypes;
            return this;
        }

        AbstractBuilder withJsonSubtypesWithNameValue(boolean jsonSubtypesWithNameValue) {
            this.jsonSubtypesWithNameValue = jsonSubtypesWithNameValue;
            return this;
        }

        public AbstractBuilder withAnnotations(List<AnnotationConf> annotations) {
            this.annotations = annotations;
            return this;
        }

        abstract AbstractModelConf build();

    }
}
