package ru.yandex.direct.model.generator.old.conf;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.typesafe.config.Config;

import ru.yandex.direct.model.generator.old.javafile.Util;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.unmodifiableSet;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringEscapeUtils.escapeJava;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.APPLY;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.PARAMS;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.PARAMS_KEY;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.PARAMS_LITERAL;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.PARAMS_VALUE;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.SourceProperty.TYPE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class AnnotationConf {
    static final String LITERAL = "$L";
    static final String STRING = "$S";

    public enum Applicability {
        FIELD,
        GETTER,
        SETTER_PARAMETER
    }

    public static class Param {
        private final String key;
        private final String formatter;
        private final String codeBlock;

        public Param(String key, String formatter, String codeBlock) {
            this.key = key;
            this.formatter = formatter;
            this.codeBlock = codeBlock;
        }

        @Override
        public String toString() {
            final StringBuilder sb = new StringBuilder("Param{");
            sb.append("key=").append(key);
            sb.append(", codeBlock='").append(codeBlock).append('\'');
            sb.append('}');
            return sb.toString();
        }

        @Override
        @SuppressWarnings("EqualsGetClass")
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Param param = (Param) o;
            return Objects.equals(key, param.key) &&
                    Objects.equals(formatter, param.formatter) &&
                    Objects.equals(codeBlock, param.codeBlock);
        }

        @Override
        public int hashCode() {
            return Objects.hash(key, formatter, codeBlock);
        }
    }

    static class SourceProperty {
        static final String TYPE = "type";
        static final String APPLY = "apply";
        static final String PARAMS = "params";
        static final String PARAMS_KEY = "key";
        static final String PARAMS_VALUE = "value";
        static final String PARAMS_LITERAL = "literal";

        private SourceProperty() {
        }
    }

    private final ClassName type;
    private final Set<Applicability> applicabilities;
    private final List<Param> params;

    private AnnotationConf(ClassName type, Set<Applicability> applicabilities,
                           List<Param> params) {
        this.type = checkNotNull(type);
        this.applicabilities = unmodifiableSet(applicabilities);
        this.params = params;
    }

    public static AnnotationConf fromConfig(Config config, @Nullable String defaultPackage) {
        Set<Applicability> applicabilities = applicabilitiesFromConfig(config);
        List<? extends Config> paramConfigs = config.hasPath(PARAMS) ?
                config.getConfigList(PARAMS) :
                Collections.emptyList();
        return of(config.getString(TYPE), defaultPackage, applicabilities,
                mapList(paramConfigs, AnnotationConf::paramFromConfig));
    }

    private static Set<Applicability> applicabilitiesFromConfig(Config config) {
        if (config.hasPath(APPLY)) {
            return config.getStringList(APPLY).stream()
                    .map(String::toUpperCase)
                    .map(Applicability::valueOf)
                    .collect(toSet());
        }

        return EnumSet.of(Applicability.FIELD, Applicability.GETTER);
    }

    private static Param paramFromConfig(Config config) {
        String key = config.getString(PARAMS_KEY);
        boolean isLiteral = config.hasPath(PARAMS_LITERAL) && config.getBoolean(PARAMS_LITERAL);
        Object valueObj = config.getAnyRef(PARAMS_VALUE);
        if (valueObj instanceof Long || valueObj instanceof Integer || valueObj instanceof Boolean) {
            return new Param(key, LITERAL, valueObj.toString());
        } else if (valueObj instanceof String) {
            if (isLiteral) {
                return new Param(key, LITERAL, (String) valueObj);
            } else {
                return new Param(key, STRING, (String) valueObj);
            }
        } else if (valueObj instanceof List) {
            Function<Object,String> toString = isLiteral ? Object::toString : AnnotationConf::arrayPartStr;
            return new Param(key, LITERAL,
                    String.format("{%s}", String.join(",", mapList((List<?>) valueObj, toString))));
        }
        throw new IllegalArgumentException(String.format("Annotation param config %s cannot be parsed", config));
    }

    private static String arrayPartStr(Object o) {
        return o instanceof String ? String.format("\"%s\"", escapeJava((String) o)) : o.toString();
    }

    public static AnnotationConf of(String type, @Nullable String defaultPackage, Applicability applicability,
                                    List<Param> params) {
        return of(type, defaultPackage, EnumSet.of(applicability), params);
    }

    public static AnnotationConf of(String type, @Nullable String defaultPackage, Set<Applicability> applicabilities,
                                    List<Param> params) {
        return new AnnotationConf(Util.classNameOf(type, defaultPackage), applicabilities, params);
    }

    public AnnotationSpec toSpec() {
        AnnotationSpec.Builder builder = AnnotationSpec.builder(type);
        params.forEach(p -> builder.addMember(p.key, p.formatter, p.codeBlock));
        return builder.build();
    }

    public boolean annotateGetter() {
        return applicabilities.contains(Applicability.GETTER);
    }

    public boolean annotateField() {
        return applicabilities.contains(Applicability.FIELD);
    }

    public boolean annotateSetterParameter() {
        return applicabilities.contains(Applicability.SETTER_PARAMETER);
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("AnnotationConf{");
        sb.append("type=").append(type);
        sb.append(", applicabilities='").append(applicabilities).append('\'');
        sb.append(", params='").append(params).append('\'');
        sb.append('}');
        return sb.toString();
    }

    @Override
    @SuppressWarnings("EqualsGetClass")
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AnnotationConf that = (AnnotationConf) o;
        return Objects.equals(type, that.type) &&
                Objects.equals(applicabilities, that.applicabilities) &&
                Objects.equals(params, that.params);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, applicabilities, params);
    }
}
