package ru.yandex.direct.model.generator.old.conf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import com.typesafe.config.Config;

import ru.yandex.direct.model.generator.old.javafile.Util;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ANNOTATIONS;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.LITERAL;
import static ru.yandex.direct.model.generator.old.conf.AnnotationConf.STRING;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.ALIAS_TO;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.COMMENT;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.JSON;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.JSON_INCLUDE;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.RELATIONSHIP;
import static ru.yandex.direct.model.generator.old.conf.AttrConf.SourceProperty.TYPE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class AttrConf {

    static class SourceProperty {
        static final String COMMENT = "comment";
        static final String JSON = "json";
        static final String JSON_INCLUDE = "jsonInclude";
        static final String NAME = "name";
        static final String TYPE = "type";
        static final String ALIAS_TO = "aliasTo";
        static final String RELATIONSHIP = "relationship";

        private SourceProperty() {
        }
    }

    private final TypeName type;
    private final List<AnnotationConf> annotations;
    private final String name;
    private final String comment;
    private final String aliasTo;
    private final RelationshipConf relationship;

    private AttrConf(TypeName type, List<AnnotationConf> annotations, String name, String comment, String aliasTo,
                     @Nullable RelationshipConf relationship) {
        this.type = checkNotNull(type);
        this.annotations = Collections.unmodifiableList(annotations);
        this.name = checkNotNull(name);
        this.comment = checkNotNull(comment);
        this.aliasTo = checkNotNull(aliasTo);
        this.relationship = relationship;
    }

    public static AttrConf fromConfig(Config config, @Nullable String defaultPackage, ClassName containingClass,
                                      @Nullable String sourceFile) {
        String comment = config.hasPath(COMMENT) ? config.getString(COMMENT) : "";
        String json = config.hasPath(JSON) ? config.getString(JSON) : "";
        String jsonInclude = config.hasPath(JSON_INCLUDE) ? config.getString(JSON_INCLUDE) : "";
        String aliasTo = config.hasPath(ALIAS_TO) ? config.getString(ALIAS_TO) : "";
        List<? extends Config> annotationConfigs =
                config.hasPath(ANNOTATIONS) ? config.getConfigList(ANNOTATIONS) : Collections.emptyList();
        List<AnnotationConf> annotations =
                mapList(annotationConfigs, c -> AnnotationConf.fromConfig(c, defaultPackage));
        RelationshipConf relationship = null;
        if (config.hasPath(RELATIONSHIP)) {
            relationship = RelationshipConf.fromConfig(config.getConfig(RELATIONSHIP), defaultPackage,
                    containingClass,
                    Util.typeNameOf(config.getString(TYPE), defaultPackage),
                    config.getString(NAME),
                    sourceFile);
        }
        return of(config.getString(NAME), config.getString(TYPE), defaultPackage, comment, json, jsonInclude, aliasTo,
                annotations, relationship);
    }

    public static AttrConf of(String name, String type, @Nullable String defaultPackage) {
        return of(name, type, defaultPackage, "", "", "");
    }

    public static AttrConf of(String name, String type, @Nullable String defaultPackage,
                              List<AnnotationConf> annotations) {
        return of(name, type, defaultPackage, "", "", "", "", annotations, null);
    }

    public static AttrConf of(String name, String type, @Nullable String defaultPackage, String comment, String json,
                              String jsonInclude) {
        return of(name, type, defaultPackage, comment, json, jsonInclude, "", Collections.emptyList(), null);
    }

    @SuppressWarnings("checkstyle:parameternumber")
    public static AttrConf of(String name, String type, @Nullable String defaultPackage, String comment, String json,
                              String jsonInclude, String aliasTo, List<AnnotationConf> annotations,
                              @Nullable RelationshipConf relationship) {
        List<AnnotationConf> allAnnotations = new ArrayList<>(annotations);

        if (!json.isEmpty()) {
            AnnotationConf.Param param = new AnnotationConf.Param("value", STRING, json);
            allAnnotations.add(AnnotationConf
                    .of(JsonProperty.class.getCanonicalName(), defaultPackage, AnnotationConf.Applicability.FIELD,
                            Collections.singletonList(param)));
        }
        if (!jsonInclude.isEmpty()) {
            AnnotationConf.Param param =
                    new AnnotationConf.Param("value", LITERAL, String.format("JsonInclude.Include.%s", jsonInclude));
            allAnnotations.add(AnnotationConf
                    .of(JsonInclude.class.getCanonicalName(), defaultPackage, AnnotationConf.Applicability.FIELD,
                            Collections.singletonList(param)));
        }

        return new AttrConf(Util.typeNameOf(type, defaultPackage), allAnnotations, name, comment, aliasTo, relationship);
    }

    public TypeName getType() {
        return type;
    }

    public List<AnnotationConf> getAnnotations() {
        return annotations;
    }

    public String getName() {
        return name;
    }

    public String getComment() {
        return comment;
    }

    public String getAliasTo() {
        return aliasTo;
    }

    public RelationshipConf getRelationship() {
        return relationship;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("AttrConf{");
        sb.append("type=").append(type);
        sb.append(", annotations='").append(annotations).append('\'');
        sb.append(", name='").append(name).append('\'');
        sb.append(", comment='").append(comment).append('\'');
        sb.append(", relationship='").append(relationship).append('\'');
        sb.append('}');
        return sb.toString();
    }

    @Override
    @SuppressWarnings("EqualsGetClass")
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AttrConf attrConf = (AttrConf) o;
        return Objects.equals(type, attrConf.type) &&
                Objects.equals(annotations, attrConf.annotations) &&
                Objects.equals(name, attrConf.name) &&
                Objects.equals(comment, attrConf.comment) &&
                Objects.equals(relationship, attrConf.relationship);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, annotations, name, comment);
    }
}
