package ru.yandex.direct.model.generator.old.conf;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.ClassName;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.COMMENT;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.PACKAGE;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.VALUES;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.VALUES_SOURCE;
import static ru.yandex.direct.model.generator.old.conf.EnumConf.SourceProperty.VALUES_TYPE;

@ParametersAreNonnullByDefault
public class EnumConf implements ModelConf, UpperLevelModelConf {

    private final String name;
    private final String comment;
    private final String valuesType;
    private final List<EnumValueConf> values;
    private final String valuesSource;
    private String sourceFileName;
    private String packageName;
    private ClassName className;

    EnumConf(String sourceFileName, String packageName, String name, String comment, String valuesType,
             List<EnumValueConf> values, String valuesSource) {
        this.sourceFileName = sourceFileName;
        this.packageName = checkNotNull(packageName);
        this.name = checkNotNull(name);
        this.comment = checkNotNull(comment);
        this.values = unmodifiableList(values);
        this.valuesType = checkNotNull(valuesType);
        this.valuesSource = checkNotNull(valuesSource);
        checkState(values.isEmpty() ^ valuesSource.isEmpty(),
                "Error in enum %s: values xor valuesSource must be defined", name);

        this.className = ClassName.get(this.packageName, this.name);
    }

    static EnumConf fromConfig(Config config, String sourceFileName) {
        String defaultPackage = config.getString(PACKAGE);
        return fromConfig(config, sourceFileName, defaultPackage);
    }

    static EnumConf fromConfig(Config config, String sourceFileName, String defaultPackage) {
        Builder builder = new Builder(config.getString(NAME))
                .withSourceFileName(sourceFileName)
                .withPackageName(defaultPackage)
                .withComment(config.hasPath(COMMENT) ? config.getString(COMMENT) : "")
                .withValuesType(config.hasPath(VALUES_TYPE) ? config.getString(VALUES_TYPE) : "");

        if (config.hasPath(VALUES)) {
            List<EnumValueConf> values = new ArrayList<>();
            // для всех значений enum-а делать объекты - иногда дорого
            // поэтому определяем на лету, что там внутри - просто строка, или объект
            for (Object configObject : config.getList(VALUES)) {
                if (configObject instanceof ConfigObject) {
                    Config valueConfig = ((ConfigObject) configObject).toConfig();
                    values.add(EnumValueConf.fromConfig(valueConfig));
                } else if (configObject instanceof ConfigValue) {
                    ConfigValue val = (ConfigValue) configObject;
                    values.add(new EnumValueConf(val.unwrapped().toString(), "", "", ""));
                } else {
                    throw new IllegalStateException("unsupported enum value: " + configObject.getClass().getSimpleName()
                            + ", in config: " + config);
                }
            }
            builder.withValues(values);
        }

        if (config.hasPath(VALUES_SOURCE)) {
            builder.withValuesSource(config.getString(VALUES_SOURCE));
        }

        return builder.build();
    }

    @Override
    public List<String> getNestedModelsAttributesFullPaths() {
        return List.of();
    }

    @Override
    public List<AttrConf> getAttrs() {
        return List.of();
    }

    @Override
    public List<String> getAttributesFullPaths() {
        return List.of();
    }

    @Override
    public List<ModelConf> getAllModelConfs() {
        return List.of(this);
    }

    @Nullable
    @Override
    public String getSourceFile() {
        return sourceFileName;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getComment() {
        return comment;
    }

    @Override
    public String getFullName() {
        return packageName + "." + name;
    }

    @Override
    public Type getType() {
        return Type.ENUM;
    }

    @Override
    public String getPackageName() {
        return packageName;
    }

    @Override
    public ClassName getClassName() {
        return className;
    }

    @Override
    public List<String> getAttrNames() {
        return emptyList();
    }

    @Override
    public List<AnnotationConf> getAnnotations() {
        return emptyList();
    }

    @Override
    public List<String> getSupersFullNames() {
        return emptyList();
    }

    @Override
    public List<String> getExtendsList() {
        return emptyList();
    }

    @Override
    public List<String> getExtendsAndImplementsFullNames() {
        return emptyList();
    }

    public String getValuesType() {
        return valuesType;
    }

    public List<EnumValueConf> getValues() {
        return values;
    }

    public String getValuesSource() {
        return valuesSource;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("EnumConf{");
        sb.append(NAME).append("='").append(name).append('\'');
        sb.append(", ").append(COMMENT).append("='").append(comment).append('\'');
        sb.append(", ").append(VALUES).append("=").append(values);
        sb.append(", ").append(VALUES_SOURCE).append("='").append(valuesSource).append('\'');
        sb.append('}');
        return sb.toString();
    }

    static class SourceProperty {
        static final String PACKAGE = "package";
        static final String COMMENT = "comment";
        static final String NAME = "name";
        static final String VALUES = "values";
        static final String VALUES_SOURCE = "valuesSource";
        static final String VALUES_TYPE = "valuesType";

        private SourceProperty() {
        }
    }

    public static class Builder {
        private String sourceFileName;
        private String packageName;
        private String name;
        private String comment = "";
        private String valuesType = "";
        private List<EnumValueConf> values = emptyList();
        private String valuesSource = "";

        public Builder(String name) {
            this.name = name;
        }

        public Builder withSourceFileName(String sourceFileName) {
            this.sourceFileName = sourceFileName;
            return this;
        }

        public Builder withPackageName(String packageName) {
            this.packageName = packageName;
            return this;
        }

        public Builder withComment(String comment) {
            this.comment = comment;
            return this;
        }

        public Builder withValuesType(String valuesType) {
            this.valuesType = valuesType;
            return this;
        }

        public Builder withValues(List<EnumValueConf> values) {
            this.values = values;
            return this;
        }

        public Builder withValuesSource(String valuesSource) {
            this.valuesSource = valuesSource;
            return this;
        }

        public EnumConf build() {
            return new EnumConf(sourceFileName, packageName, name, comment, valuesType, values, valuesSource);
        }
    }
}
