package ru.yandex.direct.model.generator.old.conf;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.squareup.javapoet.ClassName;
import com.typesafe.config.Config;

import ru.yandex.direct.model.generator.old.javafile.Util;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ANNOTATIONS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES_WITH_NAME_VALUE;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.ATTRS;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.COMMENT;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.EXTENDS;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.EXTENDS_LIST;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.READONLY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class InterfaceConf implements ModelConf {

    static class SourceProperty {
        static final String ATTRS = "attrs";
        static final String COMMENT = "comment";
        static final String EXTENDS = "extends";
        static final String EXTENDS_LIST = "extendsList";
        static final String NAME = "name";
        static final String READONLY = "readonly";

        SourceProperty() {
        }
    }

    private final String classConfigName;
    private final String sourceFileName;
    private final String packageName;
    private final String name;
    private final String comment;
    private final List<String> attrs;
    private final List<String> extendsList;
    private final boolean readOnly;
    private final List<AnnotationConf> annotations;
    private final boolean jsonSubtypes;
    private final boolean jsonSubtypesWithNameValue;

    private ClassName className;

    private InterfaceConf(String sourceFileName, String packageName, String classConfigName, String name,
                          List<String> attrs, String comment,
                          List<String> extendsList, boolean readOnly,
                          List<AnnotationConf> annotations, boolean jsonSubtypes, boolean jsonSubtypesWithNameValue) {
        this.sourceFileName = sourceFileName;
        this.classConfigName = classConfigName;
        this.packageName = checkNotNull(packageName);
        this.name = checkNotNull(name);
        this.comment = checkNotNull(comment);
        this.attrs = unmodifiableList(attrs);
        this.extendsList = unmodifiableList(extendsList);
        this.readOnly = readOnly;
        this.annotations = annotations;
        this.jsonSubtypes = jsonSubtypes;
        this.jsonSubtypesWithNameValue = jsonSubtypesWithNameValue;

        this.className = ClassName.get(this.packageName, this.name);
    }

    static InterfaceConf fromConfig(Config config, String sourceFileName, String pkg, String classConfigName) {
        return new Builder(config.getString(NAME))
                .withSource(sourceFileName)
                .withClassConfigName(classConfigName)
                .withPackageName(pkg)
                .withAttrs(config.getStringList(ATTRS))
                .withComment(config.hasPath(COMMENT) ? config.getString(COMMENT) : "")
                .withExtendsList(config.hasPath(EXTENDS) ? config.getStringList(EXTENDS) : emptyList())
                .withReadOnly(!config.hasPath(READONLY) || config.getBoolean(READONLY))
                .withJsonSubtypes(config.hasPath(JSON_SUBTYPES) && config.getBoolean(JSON_SUBTYPES))
                .withJsonSubtypesWithNameValue(config.hasPath(JSON_SUBTYPES_WITH_NAME_VALUE) && config.getBoolean(JSON_SUBTYPES_WITH_NAME_VALUE))
                .withAnnotations(Util.configObjects(config, ANNOTATIONS, c -> AnnotationConf.fromConfig(c, pkg)))
                .build();
    }

    @Override
    public String getFullName() {
        return packageName + "." + name;
    }

    @Override
    public Type getType() {
        return Type.INTERFACE;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getPackageName() {
        return packageName;
    }

    @Override
    public String getComment() {
        return comment;
    }

    @Override
    public ClassName getClassName() {
        return className;
    }

    @Override
    public List<String> getAttrNames() {
        return attrs;
    }

    @Override
    public List<AnnotationConf> getAnnotations() {
        return annotations;
    }

    @Override
    public List<String> getSupersFullNames() {
        return mapList(extendsList, this::ensureFullName);
    }

    @Override
    public List<String> getExtendsList() {
        return extendsList;
    }

    @Override
    public List<String> getExtendsAndImplementsFullNames() {
        return getSupersFullNames();
    }

    @Nullable
    @Override
    public String getSourceFile() {
        return sourceFileName;
    }

    public String getClassConfigFullName() {
        return packageName + "." + classConfigName;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public boolean isJsonSubtypes() {
        return jsonSubtypes;
    }

    public boolean isJsonSubtypesWithNameValue() {
        return jsonSubtypesWithNameValue;
    }

    private String ensureFullName(String name) {
        return isFullName(name) ? name : getPackageName() + "." + name;
    }

    private boolean isFullName(String name) {
        return name.contains(".");
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("InterfaceConf{");
        sb.append(NAME).append("='").append(name).append('\'');
        sb.append(", ").append(COMMENT).append("='").append(comment).append('\'');
        sb.append(", ").append(ATTRS).append("='").append(attrs).append('\'');
        sb.append(", ").append(EXTENDS_LIST).append("='").append(extendsList).append('\'');
        sb.append(", ").append(READONLY).append("='").append(readOnly).append('\'');
        sb.append(", ").append(ANNOTATIONS).append("='").append(annotations).append('\'');
        sb.append(", ").append(JSON_SUBTYPES).append("='").append(jsonSubtypes).append('\'');
        sb.append(", ").append(JSON_SUBTYPES_WITH_NAME_VALUE).append("='").append(jsonSubtypesWithNameValue).append('\'');
        sb.append('}');
        return sb.toString();
    }

    public static class Builder {
        private String name;
        private String sourceFileName;
        private String packageName;
        String classConfigName;
        private List<String> attrs;
        private List<String> extendsList = emptyList();
        private String comment = "";
        private boolean readOnly = true;
        private List<AnnotationConf> annotations;
        private boolean jsonSubtypes = false;
        private boolean jsonSubtypesWithNameValue = false;

        public Builder withJsonSubtypes(boolean jsonSubtypes) {
            this.jsonSubtypes = jsonSubtypes;
            return this;
        }

        public Builder withJsonSubtypesWithNameValue(boolean jsonSubtypesWithNameValue) {
            this.jsonSubtypesWithNameValue = jsonSubtypesWithNameValue;
            return this;
        }

        public Builder withAnnotations(List<AnnotationConf> annotations) {
            this.annotations = annotations;
            return this;
        }

        public Builder(String name) {
            this.name = name;
        }

        public Builder withAttrs(List<String> attrs) {
            this.attrs = attrs;
            return this;
        }

        public Builder withComment(String comment) {
            this.comment = comment;
            return this;
        }

        public Builder withExtendsList(List<String> extendsList) {
            this.extendsList = extendsList;
            return this;
        }

        public Builder withSource(String sourceFileName) {
            this.sourceFileName = sourceFileName;
            return this;
        }

        public Builder withPackageName(String packageName) {
            this.packageName = packageName;
            return this;
        }

        public Builder withClassConfigName(String classConfigName) {
            this.classConfigName = classConfigName;
            return this;
        }

        public Builder withReadOnly(boolean readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public InterfaceConf build() {
            return new InterfaceConf(sourceFileName, packageName, classConfigName, name, attrs, comment, extendsList,
                    readOnly, annotations, jsonSubtypes, jsonSubtypesWithNameValue);
        }

        public String getName() {
            return name;
        }

        List<String> getAttrs() {
            return attrs;
        }

        String getComment() {
            return comment;
        }
    }
}
