package ru.yandex.direct.model.generator.old.conf;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.lang.model.element.Modifier;

import com.google.common.collect.ImmutableList;
import com.squareup.javapoet.ClassName;
import com.typesafe.config.Config;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.javafile.Util;
import ru.yandex.direct.model.generator.old.util.AttrNameUtil;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ANNOTATIONS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ATTRS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.COMMENT;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ENUMS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.EXTENDS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.GENERATE_COPY_METHOD;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.GENERATE_PROPERTIES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.IMPLEMENTS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.INTERFACES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES_WITH_NAME_VALUE;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.MODIFIERS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.PACKAGE;
import static ru.yandex.direct.model.generator.old.conf.ModelConf.Type.CLASS;

@ParametersAreNonnullByDefault
public class ModelClassConf extends AbstractModelConf implements UpperLevelModelConf {

    private final String extendsClass;
    private final List<Modifier> modifiers;
    private final List<EnumConf> enums;
    private final List<InterfaceConf> interfaces;
    private final List<String> implementsList;
    private final boolean generateCopyMethod;

    private ClassName className;

    @SuppressWarnings("squid:S00107")   // приватный конструктор, делаем что хотим
    private ModelClassConf(String sourceFileName, String packageName, String name, String extendsClass,
                           List<Modifier> modifiers, String comment,
                           List<AttrConf> attrs, List<AnnotationConf> annotations, List<EnumConf> enums,
                           List<InterfaceConf> interfaces, List<String> implementsList, boolean generateProperties,
                           boolean generateCopyMethod, boolean jsonSubtypes, boolean jsonSubtypesWithNameValue) {
        super(sourceFileName, packageName, name, comment, attrs, annotations, generateProperties, jsonSubtypes, jsonSubtypesWithNameValue);
        this.extendsClass = checkNotNull(extendsClass);
        this.modifiers = modifiers;
        this.enums = unmodifiableList(enums);
        this.interfaces = unmodifiableList(interfaces);
        this.implementsList = implementsList;
        this.generateCopyMethod = generateCopyMethod;
        this.className = ClassName.get(packageName, name);
    }

    public static ModelClassConf fromConfig(Config config, String sourceFileName) {
        String pkg = config.getString(PACKAGE);
        String classConfigName = config.getString(NAME);
        ClassName className = ClassName.get(pkg, config.getString(NAME));

        Builder builder = new Builder(pkg, config.getString(NAME))
                .withExtendsClass(config.hasPath(EXTENDS) ? config.getString(EXTENDS) : "");
        if (config.hasPath(MODIFIERS)) {
            builder.withModifiers(config.getStringList(MODIFIERS).stream().map(Modifier::valueOf).collect(toList()));
        }
        //noinspection SimplifiableConditionalExpression
        builder.withComment(config.hasPath(COMMENT) ? config.getString(COMMENT) : "")
                .withSource(sourceFileName)
                .withGenerateProperties(
                        config.hasPath(GENERATE_PROPERTIES)
                                ? config.getBoolean(GENERATE_PROPERTIES)
                                : true)
                .withGenerateCopyMethod(
                        config.hasPath(GENERATE_COPY_METHOD)
                                ? config.getBoolean(GENERATE_COPY_METHOD)
                                : false)
                .withJsonSubtypes(
                        config.hasPath(JSON_SUBTYPES)
                                ? config.getBoolean(JSON_SUBTYPES)
                                : false)
                .withJsonSubtypesWithNameValue(
                        config.hasPath(JSON_SUBTYPES_WITH_NAME_VALUE)
                                ? config.getBoolean(JSON_SUBTYPES_WITH_NAME_VALUE)
                                : false)
                .withAnnotations(Util.configObjects(config, ANNOTATIONS, c -> AnnotationConf.fromConfig(c, pkg)))
                .withEnums(Util.configObjects(config, ENUMS, c -> EnumConf.fromConfig(c, sourceFileName, pkg)))
                .withAttrs(Util.configObjects(config, ATTRS, c -> AttrConf.fromConfig(c, pkg, className,
                        sourceFileName)))
                .withInterfaces(
                        Util.configObjects(config, INTERFACES, c -> InterfaceConf.fromConfig(c, sourceFileName, pkg,
                                classConfigName)))
                .withImplementsList(config.hasPath(IMPLEMENTS) ? config.getStringList(IMPLEMENTS) : emptyList());

        return builder.build();
    }

    public String getExtendsClass() {
        return extendsClass;
    }

    public List<Modifier> getModifiers() {
        return modifiers;
    }

    private List<EnumConf> getEnums() {
        return enums;
    }

    public List<InterfaceConf> getInterfaces() {
        return interfaces;
    }

    private List<RelationshipConf> getRelationships() {
        return StreamEx.of(getAttrs()).map(AttrConf::getRelationship).nonNull().toList();
    }

    public List<String> getImplementsList() {
        return implementsList;
    }

    public boolean isGenerateCopyMethod() {
        return generateCopyMethod;
    }

    @Override
    public List<String> getSupersFullNames() {
        return singletonList(getExtendsClassFullName());
    }

    @Override
    public List<String> getExtendsAndImplementsFullNames() {
        Set<String> set = getImplementsFullNames();
        set.addAll(getNestedInterfacesFullNames());
        set.add(getExtendsClassFullName());
        return new ArrayList<>(set);
    }

    @Override
    public List<String> getNestedModelsAttributesFullPaths() {
        return StreamEx.of(interfaces)
                .mapToEntry(InterfaceConf::getName, InterfaceConf::getAttrNames)
                .mapKeys(this::ensureFullName)
                .flatMapValues(Collection::stream)
                .mapKeyValue((interfaceFullName, attrName) -> AttrNameUtil
                        .getAttrFullName(attrName, () -> interfaceFullName))
                .toList();

    }

    private String getExtendsClassFullName() {
        return ensureFullName(extendsClass);
    }

    private Set<String> getImplementsFullNames() {
        return implementsList.stream()
                .map(this::ensureFullName)
                .collect(toSet());
    }

    private Set<String> getNestedInterfacesFullNames() {
        return interfaces.stream()
                .map(InterfaceConf::getName)
                .map(this::ensureFullName)
                .collect(toSet());
    }

    private String ensureFullName(String name) {
        return isFullName(name) ? name : getPackageName() + "." + name;
    }

    private boolean isFullName(String name) {
        return name.contains(".");
    }

    @Override
    public List<String> getExtendsList() {
        return singletonList(getExtendsClass());
    }

    @Override
    public Type getType() {
        return CLASS;
    }

    @Override
    public List<ModelConf> getAllModelConfs() {
        return ImmutableList.<ModelConf>builder()
                .add(this)
                .addAll(getInterfaces())
                .addAll(getEnums())
                .addAll(getRelationships())
                .build();
    }

    @Override
    public String toString() {
        return "ModelClassConf{" + "extendsClass='" + extendsClass + '\'' +
                ", modifiers=" + modifiers +
                ", enums=" + enums +
                ", interfaces=" + interfaces +
                ", relationships=" + getRelationships() +
                ", implementsList=" + implementsList +
                ", className=" + className +
                ", supersFullNames=" + getSupersFullNames() +
                ", extendsAndImplementsFullNames=" + getExtendsAndImplementsFullNames() +
                ", extendsList=" + getExtendsList() +
                ", type=" + getType() +
                '}';
    }

    @Override
    @SuppressWarnings("EqualsGetClass")
    public boolean equals(@Nullable Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        ModelClassConf classConf = (ModelClassConf) o;
        return java.util.Objects.equals(extendsClass, classConf.extendsClass) &&
                java.util.Objects.equals(modifiers, classConf.modifiers) &&
                java.util.Objects.equals(enums, classConf.enums) &&
                java.util.Objects.equals(interfaces, classConf.interfaces) &&
                java.util.Objects.equals(implementsList, classConf.implementsList) &&
                java.util.Objects.equals(className, classConf.className);
    }

    @Override
    public int hashCode() {
        return java.util.Objects
                .hash(super.hashCode(), extendsClass, modifiers, enums, interfaces, implementsList, className);
    }

    public static class Builder extends AbstractBuilder {
        private String extendsClass = "";
        private List<Modifier> modifiers = emptyList();
        private List<EnumConf> enums = emptyList();
        private List<InterfaceConf> interfaces = emptyList();
        private List<String> implementsList = emptyList();
        private boolean generateCopyMethod = false;

        public Builder(String pack, String name) {
            super(pack, name);
        }

        boolean isGenerateCopyMethod() {
            return generateCopyMethod;
        }

        public Builder withExtendsClass(String extendsClass) {
            this.extendsClass = extendsClass;
            return this;
        }

        public Builder withModifiers(List<Modifier> modifiers) {
            this.modifiers = modifiers;
            return this;
        }

        @Override
        public Builder withSource(String sourceFileName) {
            super.withSource(sourceFileName);
            return this;
        }

        @Override
        public Builder withComment(String comment) {
            super.withComment(comment);
            return this;
        }

        @Override
        public Builder withAttrs(List<AttrConf> attrs) {
            super.withAttrs(attrs);
            return this;
        }

        public Builder withEnums(List<EnumConf> enums) {
            this.enums = enums;
            return this;
        }

        @Override
        public Builder withAnnotations(List<AnnotationConf> annotations) {
            super.withAnnotations(annotations);
            return this;
        }

        public Builder withInterfaces(List<InterfaceConf> interfaces) {
            this.interfaces = interfaces;
            return this;
        }

        public Builder withImplementsList(List<String> implementsList) {
            this.implementsList = implementsList;
            return this;
        }

        @Override
        public Builder withJsonSubtypes(boolean jsonSubtypes) {
            super.withJsonSubtypes(jsonSubtypes);
            return this;
        }

        @Override
        public Builder withJsonSubtypesWithNameValue(boolean jsonSubtypesWithNameValue) {
            super.withJsonSubtypesWithNameValue(jsonSubtypesWithNameValue);
            return this;
        }

        @Override
        Builder withGenerateProperties(boolean generateProperties) {
            super.withGenerateProperties(generateProperties);
            return this;
        }

        public Builder withGenerateCopyMethod(boolean generateCopyMethod) {
            this.generateCopyMethod = generateCopyMethod;
            return this;
        }


        @Override
        public ModelClassConf build() {
            return new ModelClassConf(getSource(), getPack(), getName(), extendsClass, modifiers, getComment(),
                    getAttrs(), getAnnotations(), enums, interfaces, implementsList, isGenerateProperties(),
                    isGenerateCopyMethod(), isJsonSubtypes(), isJsonSubtypesWithNameValue());
        }
    }
}
