package ru.yandex.direct.model.generator.old.conf;

import java.io.File;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.PathMatcher;

import javax.annotation.ParametersAreNonnullByDefault;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

@ParametersAreNonnullByDefault
public class ModelConfFactory {

    private static final String SA_INTERFACE_FILE_NAME_PATTERN = "**/i_*.conf";
    private static final String SA_RELATIONSHIP_FILE_NAME_PATTERN = "**/r_*.conf";
    private static final String SA_ENUM_FILE_NAME_PATTERN = "**/e_*.conf";

    private static final PathMatcher INTERFACE_PATH_MATCHER = FileSystems.getDefault()
            .getPathMatcher("glob:" + SA_INTERFACE_FILE_NAME_PATTERN);
    private static final PathMatcher RELATIONSHIP_PATH_MATCHER = FileSystems.getDefault()
            .getPathMatcher("glob:" + SA_RELATIONSHIP_FILE_NAME_PATTERN);
    private static final PathMatcher ENUM_PATH_MATCHER = FileSystems.getDefault()
            .getPathMatcher("glob:" + SA_ENUM_FILE_NAME_PATTERN);

    /**
     * Создаёт описание модели верхнего уровня на основе исходного файла.
     *
     * @param source *.conf-исходник. Если имя файла соответствует паттерну
     *               {@value #SA_INTERFACE_FILE_NAME_PATTERN},
     *               он будет парситься как {@link ModelInterfaceConf};
     *               {@value #SA_RELATIONSHIP_FILE_NAME_PATTERN} — то как {@link RelationshipConf};
     *               в противном случае – как {@link ModelClassConf}.
     * @return Описание модели верхнего уровня, инстанс {@link UpperLevelModelConf}.
     */
    public UpperLevelModelConf createModelConf(URL source) {
        Config config = ConfigFactory.parseURL(source);
        String sourceFileName = source.getFile();
        File file = new File(sourceFileName);
        if (isStandaloneInterfaceConfig(file)) {
            return ModelInterfaceConf.fromConfig(config, sourceFileName);
        } else if (isStandaloneRelationshipConfig(file)) {
            return RelationshipConf.fromConfig(config, sourceFileName);
        } else if (isStandaloneEnumConfig(file)) {
            return EnumConf.fromConfig(config, sourceFileName);
        } else {
            return ModelClassConf.fromConfig(config, sourceFileName);
        }
    }

    /**
     * Является ли данный файл описанием интерфейса типа {@link ModelInterfaceConf}.
     * Определяется на основе соглашения, что все файлы отдельно стоящих описаний интерфейсов
     * должны начинаться с "i_".
     *
     * @param source исходный файл.
     * @return {@code true}, если имя файла соответствует паттерну {@value SA_INTERFACE_FILE_NAME_PATTERN}.
     */
    private static boolean isStandaloneInterfaceConfig(File source) {
        return INTERFACE_PATH_MATCHER.matches(source.toPath());
    }

    private static boolean isStandaloneRelationshipConfig(File source) {
        return RELATIONSHIP_PATH_MATCHER.matches(source.toPath());
    }

    private static boolean isStandaloneEnumConfig(File source) {
        return ENUM_PATH_MATCHER.matches(source.toPath());
    }

}
