package ru.yandex.direct.model.generator.old.conf;

import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.squareup.javapoet.ClassName;
import com.typesafe.config.Config;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.generator.old.javafile.Util;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ANNOTATIONS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ATTRS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.COMMENT;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.ENUMS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.EXTENDS;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.GENERATE_PROPERTIES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.JSON_SUBTYPES_WITH_NAME_VALUE;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.NAME;
import static ru.yandex.direct.model.generator.old.conf.AbstractModelConf.SourceProperty.PACKAGE;
import static ru.yandex.direct.model.generator.old.conf.InterfaceConf.SourceProperty.READONLY;
import static ru.yandex.direct.model.generator.old.conf.ModelConf.Type.INTERFACE;

/**
 * Описание интерфейса из отдельного .conf файла.
 */
@ParametersAreNonnullByDefault
public class ModelInterfaceConf extends AbstractModelConf implements UpperLevelModelConf {

    private final List<String> extendsList;
    private final List<EnumConf> enums;
    private final boolean readonly;

    @SuppressWarnings("squid:S00107")   // приватный конструктор, делаем что хотим
    private ModelInterfaceConf(String sourceFileName, String packageName, String name, List<AttrConf> attrs,
                               List<AnnotationConf> annotations, String comment,
                               List<String> extendsList, boolean generateProperties, boolean jsonSubtypes,
                               boolean jsonSubtypesWithNameValue, List<EnumConf> enums, boolean readonly) {
        super(sourceFileName, packageName, name, comment, attrs, annotations, generateProperties, jsonSubtypes, jsonSubtypesWithNameValue);
        this.extendsList = unmodifiableList(extendsList);
        this.enums = unmodifiableList(enums);
        this.readonly = readonly;
    }

    static ModelInterfaceConf fromConfig(Config config, String sourceFileName) {
        String pkg = config.getString(PACKAGE);
        ClassName interfaceName = ClassName.get(pkg, config.getString(NAME));
        //noinspection SimplifiableConditionalExpression
        return new Builder(pkg, config.getString(NAME))
                .withSource(sourceFileName)
                .withComment(config.hasPath(COMMENT) ? config.getString(COMMENT) : "")
                .withGenerateProperties(
                        config.hasPath(GENERATE_PROPERTIES)
                                ? config.getBoolean(GENERATE_PROPERTIES)
                                : true)
                .withJsonSubtypes(
                        config.hasPath(JSON_SUBTYPES)
                                ? config.getBoolean(JSON_SUBTYPES)
                                : false)
                .withJsonSubtypesWithNameValue(
                        config.hasPath(JSON_SUBTYPES_WITH_NAME_VALUE)
                                ? config.getBoolean(JSON_SUBTYPES_WITH_NAME_VALUE)
                                : false)
                .withReadonly(!config.hasPath(READONLY) || config.getBoolean(READONLY))
                .withAttrs(Util.configObjects(config, ATTRS,
                        c -> AttrConf.fromConfig(c, pkg, interfaceName, sourceFileName)))
                .withAnnotations(Util.configObjects(config, ANNOTATIONS, c -> AnnotationConf.fromConfig(c, pkg)))
                .withExtendsList(config.hasPath(EXTENDS) ? config.getStringList(EXTENDS) : emptyList())
                .withEnums(Util.configObjects(config, ENUMS, c -> EnumConf.fromConfig(c, sourceFileName, pkg)))
                .build();
    }

    @Override
    public List<String> getExtendsList() {
        return extendsList;
    }

    @Override
    public List<String> getSupersFullNames() {
        return getExtendsFullNames();
    }

    public List<EnumConf> getEnums() {
        return enums;
    }

    public List<RelationshipConf> getRelationships() {
        return StreamEx.of(getAttrs()).map(AttrConf::getRelationship).nonNull().toList();
    }

    @Override
    public List<String> getExtendsAndImplementsFullNames() {
        return getSupersFullNames();
    }

    public boolean isReadonly() {
        return readonly;
    }

    private List<String> getExtendsFullNames() {
        return extendsList.stream()
                .map(s -> getPackageName() + "." + s)
                .collect(toList());
    }

    @Override
    public Type getType() {
        return INTERFACE;
    }

    @Override
    public List<ModelConf> getAllModelConfs() {
        return ImmutableList.<ModelConf>builder()
                .add(this)
                .addAll(getEnums())
                .addAll(getRelationships())
                .build();
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("ModelInterfaceConf{");
        sb.append("extendsList=").append(extendsList);
        sb.append(", supersFullNames=").append(getSupersFullNames());
        sb.append(", extendsAndImplementsFullNames=").append(getExtendsAndImplementsFullNames());
        sb.append(", type=").append(getType());
        sb.append(", jsonSubtypes=").append(isJsonSubtypes());
        sb.append(", jsonSubtypesWithNameValue=").append(isJsonSubtypesWithNameValue());
        sb.append(", enums=").append(enums);
        sb.append('}');
        return sb.toString();
    }

    @Override
    @SuppressWarnings("EqualsGetClass")
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        ModelInterfaceConf that = (ModelInterfaceConf) o;
        return Objects.equals(enums, that.enums) &&
                Objects.equals(extendsList, that.extendsList);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), extendsList, enums);
    }

    public static class Builder extends AbstractBuilder {

        private boolean readonly = true;
        private List<String> extendsList;
        private List<EnumConf> enums = emptyList();

        Builder(String packageName, String name) {
            super(packageName, name);
        }

        Builder withExtendsList(List<String> extendsList) {
            this.extendsList = extendsList;
            return this;
        }

        @Override
        Builder withSource(String sourceFileName) {
            super.withSource(sourceFileName);
            return this;
        }

        @Override
        public Builder withComment(String comment) {
            super.withComment(comment);
            return this;
        }

        @Override
        public Builder withAttrs(List<AttrConf> attrs) {
            super.withAttrs(attrs);
            return this;
        }

        @Override
        public Builder withAnnotations(List<AnnotationConf> annotations) {
            super.withAnnotations(annotations);
            return this;
        }

        @Override
        public Builder withGenerateProperties(boolean generateProperties) {
            super.withGenerateProperties(generateProperties);
            return this;
        }

        @Override
        public Builder withJsonSubtypes(boolean jsonSubtypes) {
            super.withJsonSubtypes(jsonSubtypes);
            return this;
        }

        @Override
        public Builder withJsonSubtypesWithNameValue(boolean jsonSubtypesWithNameValue) {
            super.withJsonSubtypesWithNameValue(jsonSubtypesWithNameValue);
            return this;
        }

        public Builder withReadonly(boolean readonly) {
            this.readonly = readonly;
            return this;
        }

        public Builder withEnums(List<EnumConf> enums) {
            this.enums = enums;
            return this;
        }

        @Override
        public ModelInterfaceConf build() {
            return new ModelInterfaceConf(getSource(), getPack(), getName(),
                    getAttrs(), getAnnotations(), getComment(), extendsList, isGenerateProperties(), isJsonSubtypes(),
                    isJsonSubtypesWithNameValue(), enums, readonly);
        }
    }
}
